// PagSeguro Service - API Integration

const axios = require('axios');
const crypto = require('crypto');

class PagSeguroService {
    constructor(email, token, sandbox = true) {
        this.email = email;
        this.token = token;
        this.sandbox = sandbox;
        
        this.baseUrl = sandbox ? 
            'https://ws.sandbox.pagseguro.uol.com.br' :
            'https://ws.pagseguro.uol.com.br';
            
        this.checkoutUrl = sandbox ?
            'https://sandbox.pagseguro.uol.com.br' :
            'https://pagseguro.uol.com.br';
    }

    async createPayment(data) {
        try {
            const paymentData = {
                email: this.email,
                token: this.token,
                currency: 'BRL',
                itemId1: '1',
                itemDescription1: data.description || 'Payment via WhatsApp',
                itemAmount1: data.amount.toFixed(2),
                itemQuantity1: '1',
                reference: data.reference_id || Date.now().toString(),
                senderName: data.customer_name || 'Customer',
                senderAreaCode: this.getAreaCode(data.customer_phone),
                senderPhone: this.formatPhone(data.customer_phone),
                redirectURL: `${process.env.BASE_URL || 'http://localhost:7000'}/payments/success`,
                maxAge: 86400, // 24 hours
                maxUses: 1
            };

            const response = await axios.post(
                `${this.baseUrl}/v2/checkout`,
                new URLSearchParams(paymentData).toString(),
                {
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded'
                    }
                }
            );

            // Parse XML response
            const code = this.extractCodeFromXML(response.data);
            
            if (code) {
                return {
                    success: true,
                    payment_id: code,
                    payment_url: `${this.checkoutUrl}/v2/checkout/payment.html?code=${code}`,
                    status: 'pending'
                };
            } else {
                throw new Error('Checkout code not found in response');
            }
        } catch (error) {
            console.error('Error creating PagSeguro payment:', error);
            
            let errorMessage = 'Error creating PagSeguro payment';
            
            if (error.response?.data) {
                // Try to extract error message from XML
                const errorMatch = error.response.data.match(/<message>(.*?)<\/message>/);
                if (errorMatch) {
                    errorMessage = errorMatch[1];
                }
            } else if (error.message) {
                errorMessage = error.message;
            }
            
            return {
                success: false,
                error: errorMessage,
                details: error.response?.data || null
            };
        }
    }

    async getPaymentStatus(transactionCode) {
        try {
            const response = await axios.get(
                `${this.baseUrl}/v3/transactions/${transactionCode}`,
                {
                    params: {
                        email: this.email,
                        token: this.token
                    }
                }
            );

            // Parse XML response
            const status = this.parseTransactionStatus(response.data);
            
            return {
                success: true,
                status: status.status,
                paid_at: status.paid_at,
                pagseguro_status: status.original_status
            };
        } catch (error) {
            console.error('Error checking PagSeguro status:', error);
            return {
                success: false,
                error: error.message || 'Error checking status'
            };
        }
    }

    async getTransactionByNotification(notificationCode) {
        try {
            const response = await axios.get(
                `${this.baseUrl}/v3/transactions/notifications/${notificationCode}`,
                {
                    params: {
                        email: this.email,
                        token: this.token
                    }
                }
            );

            // Parse XML response
            const transaction = this.parseTransactionXML(response.data);
            
            return {
                success: true,
                transaction: transaction
            };
        } catch (error) {
            console.error('Error fetching PagSeguro transaction:', error);
            return {
                success: false,
                error: error.message || 'Error fetching transaction'
            };
        }
    }

    // Utilities for XML parsing
    extractCodeFromXML(xmlString) {
        const match = xmlString.match(/<code>(.*?)<\/code>/);
        return match ? match[1] : null;
    }

    parseTransactionStatus(xmlString) {
        const statusMatch = xmlString.match(/<status>(.*?)<\/status>/);
        const dateMatch = xmlString.match(/<lastEventDate>(.*?)<\/lastEventDate>/);
        
        const originalStatus = statusMatch ? parseInt(statusMatch[1]) : 0;
        
        // Map PagSeguro status to our system
        let status = 'pending';
        let paid_at = null;
        
        switch (originalStatus) {
            case 1: // Awaiting payment
                status = 'pending';
                break;
            case 2: // Under review
                status = 'pending';
                break;
            case 3: // Paid
                status = 'paid';
                paid_at = dateMatch ? new Date(dateMatch[1]) : new Date();
                break;
            case 4: // Available
                status = 'paid';
                paid_at = dateMatch ? new Date(dateMatch[1]) : new Date();
                break;
            case 5: // In dispute
                status = 'pending';
                break;
            case 6: // Refunded
                status = 'cancelled';
                break;
            case 7: // Cancelled
                status = 'cancelled';
                break;
            default:
                status = 'pending';
        }
        
        return {
            status: status,
            paid_at: paid_at,
            original_status: originalStatus
        };
    }

    parseTransactionXML(xmlString) {
        const codeMatch = xmlString.match(/<code>(.*?)<\/code>/);
        const referenceMatch = xmlString.match(/<reference>(.*?)<\/reference>/);
        const statusMatch = xmlString.match(/<status>(.*?)<\/status>/);
        const dateMatch = xmlString.match(/<lastEventDate>(.*?)<\/lastEventDate>/);
        
        return {
            code: codeMatch ? codeMatch[1] : null,
            reference: referenceMatch ? referenceMatch[1] : null,
            status: statusMatch ? parseInt(statusMatch[1]) : 0,
            date: dateMatch ? dateMatch[1] : null
        };
    }

    formatPhone(phone) {
        // Remove non-numeric characters
        const cleanPhone = phone.replace(/\D/g, '');
        
        // Format expected by PagSeguro: number without area code
        // Area code must be sent separately in senderAreaCode field
        if (cleanPhone.length === 11) {
            // Mobile: (XX) 9XXXX-XXXX
            return cleanPhone.substring(2);
        } else if (cleanPhone.length === 10) {
            // Landline: (XX) XXXX-XXXX
            return cleanPhone.substring(2);
        } else if (cleanPhone.length === 13 && cleanPhone.startsWith('55')) {
            // With country code +55
            return cleanPhone.substring(4);
        }
        
        return cleanPhone;
    }

    getAreaCode(phone) {
        // Extract area code from phone
        const cleanPhone = phone.replace(/\D/g, '');
        
        if (cleanPhone.length >= 10) {
            if (cleanPhone.startsWith('55') && cleanPhone.length === 13) {
                // With country code +55
                return cleanPhone.substring(2, 4);
            }
            // Area code in first 2 digits
            return cleanPhone.substring(0, 2);
        }
        
        return '11'; // Default area code if unable to extract
    }

    generateNotificationHash(notificationCode) {
        // Generate hash for notification validation
        return crypto
            .createHash('sha256')
            .update(notificationCode + this.token)
            .digest('hex');
    }
}

module.exports = PagSeguroService;
