const express = require('express');
const webpush = require('web-push');
const { pool } = require('../config/database');
const { authenticateToken } = require('../middleware/auth');

const router = express.Router();

// Configure VAPID keys for web push
const vapidKeys = {
  publicKey: process.env.VAPID_PUBLIC_KEY || 'BEl62iUYgUivxIkv69yViEuiBIa40HI2BN4EMYn5-Fp2rtBGKh_dQbVFf9-VqJzQoOJM6A7wjuO6zSjJ1Rk',
  privateKey: process.env.VAPID_PRIVATE_KEY || 'UzxgMTfFNUriAX7ycoHiLdpzEFI7QGAjWfqtBBdagBs'
};

webpush.setVapidDetails(
  'mailto:admin@ai.com',
  vapidKeys.publicKey,
  vapidKeys.privateKey
);

// ===== SUBSCRIPTION MANAGEMENT =====

// Register new subscription
router.post('/subscribe', authenticateToken, async (req, res) => {
  let connection;
  try {
    const { subscription, userId, deviceInfo } = req.body;
    
    if (!subscription || !subscription.endpoint) {
      return res.status(400).json({ error: 'Invalid subscription' });
    }

    connection = await pool.getConnection();
    
    // Check if subscription already exists for this user/device
    const [existing] = await connection.execute(
      'SELECT id FROM push_subscriptions WHERE user_id = ? AND endpoint = ?',
      [userId || req.user.id, subscription.endpoint]
    );

    if (existing.length > 0) {
      // Update existing subscription
      await connection.execute(`
        UPDATE push_subscriptions 
        SET subscription_data = ?, device_info = ?, updated_at = CURRENT_TIMESTAMP, active = TRUE
        WHERE id = ?
      `, [
        JSON.stringify(subscription),
        JSON.stringify(deviceInfo || {}),
        existing[0].id
      ]);
    } else {
      // Create new subscription
      await connection.execute(`
        INSERT INTO push_subscriptions (user_id, endpoint, subscription_data, device_info, active)
        VALUES (?, ?, ?, ?, TRUE)
      `, [
        userId || req.user.id,
        subscription.endpoint,
        JSON.stringify(subscription),
        JSON.stringify(deviceInfo || {})
      ]);
    }

    res.json({ 
      success: true, 
      message: 'Push notifications activated successfully',
      vapidPublicKey: vapidKeys.publicKey
    });
  } catch (error) {
    console.error('Error registering subscription:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Cancel subscription
router.post('/unsubscribe', authenticateToken, async (req, res) => {
  let connection;
  try {
    const { endpoint } = req.body;
    
    connection = await pool.getConnection();
    
    await connection.execute(
      'UPDATE push_subscriptions SET active = FALSE WHERE endpoint = ? AND user_id = ?',
      [endpoint, req.user.id]
    );

    res.json({ 
      success: true, 
      message: 'Push notifications deactivated' 
    });
  } catch (error) {
    console.error('Error cancelling subscription:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// List user subscriptions
router.get('/subscriptions', authenticateToken, async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    
    const [subscriptions] = await connection.execute(`
      SELECT 
        id, endpoint, device_info, active, created_at, updated_at
      FROM push_subscriptions 
      WHERE user_id = ? AND active = TRUE
      ORDER BY updated_at DESC
    `, [req.user.id]);

    res.json(subscriptions);
  } catch (error) {
    console.error('Error fetching subscriptions:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== NOTIFICATION SENDING =====

// Send notification to specific user
router.post('/send', authenticateToken, async (req, res) => {
  let connection;
  try {
    const { 
      userId, 
      title, 
      body, 
      icon, 
      badge,
      tag,
      data,
      actions,
      requireInteraction = true,
      silent = false
    } = req.body;

    if (!userId || !title || !body) {
      return res.status(400).json({ 
        error: 'userId, title and body are required' 
      });
    }

    connection = await pool.getConnection();
    
    // Fetch user's active subscriptions
    const [subscriptions] = await connection.execute(
      'SELECT subscription_data FROM push_subscriptions WHERE user_id = ? AND active = TRUE',
      [userId]
    );

    if (subscriptions.length === 0) {
      return res.status(404).json({ 
        error: 'User does not have push notifications enabled' 
      });
    }

    // Prepare notification payload
    const notificationPayload = {
      title,
      body,
      icon: icon || '/images/icon-192x192.png',
      badge: badge || '/images/badge-72x72.png',
      tag: tag || 'default',
      requireInteraction,
      silent,
      data: {
        timestamp: Date.now(),
        url: '/user',
        ...data
      }
    };

    if (actions && Array.isArray(actions)) {
      notificationPayload.actions = actions;
    }

    // Send to all user subscriptions
    const results = [];
    
    for (const sub of subscriptions) {
      try {
        const subscription = JSON.parse(sub.subscription_data);
        
        const result = await webpush.sendNotification(
          subscription,
          JSON.stringify(notificationPayload)
        );
        
        results.push({
          endpoint: subscription.endpoint,
          success: true,
          statusCode: result.statusCode
        });
      } catch (error) {
        console.error('Error sending notification:', error);
        
        results.push({
          endpoint: subscription.endpoint,
          success: false,
          error: error.message
        });
        
        // If subscription is invalid, deactivate
        if (error.statusCode === 410) {
          await connection.execute(
            'UPDATE push_subscriptions SET active = FALSE WHERE subscription_data = ?',
            [sub.subscription_data]
          );
        }
      }
    }

    // Save notification log
    await connection.execute(`
      INSERT INTO notification_logs (user_id, title, body, payload, results, sent_at)
      VALUES (?, ?, ?, ?, ?, CURRENT_TIMESTAMP)
    `, [
      userId,
      title,
      body,
      JSON.stringify(notificationPayload),
      JSON.stringify(results)
    ]);

    const successCount = results.filter(r => r.success).length;
    
    res.json({
      success: true,
      message: `Notification sent to ${successCount}/${results.length} devices`,
      results
    });
  } catch (error) {
    console.error('Error sending notification:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Send mass notification
router.post('/broadcast', authenticateToken, async (req, res) => {
  let connection;
  try {
    const { 
      title, 
      body, 
      icon, 
      badge,
      tag,
      data,
      actions,
      userIds, // Array of specific IDs or null for all
      requireInteraction = true,
      silent = false
    } = req.body;

    if (!title || !body) {
      return res.status(400).json({ 
        error: 'title and body are required' 
      });
    }

    connection = await pool.getConnection();
    
    // Fetch subscriptions based on criteria
    let query = 'SELECT user_id, subscription_data FROM push_subscriptions WHERE active = TRUE';
    let params = [];
    
    if (userIds && Array.isArray(userIds) && userIds.length > 0) {
      query += ` AND user_id IN (${userIds.map(() => '?').join(',')})`;
      params = userIds;
    }
    
    const [subscriptions] = await connection.execute(query, params);

    if (subscriptions.length === 0) {
      return res.status(404).json({ 
        error: 'No active subscriptions found' 
      });
    }

    // Prepare notification payload
    const notificationPayload = {
      title,
      body,
      icon: icon || '/images/icon-192x192.png',
      badge: badge || '/images/badge-72x72.png',
      tag: tag || 'broadcast',
      requireInteraction,
      silent,
      data: {
        timestamp: Date.now(),
        url: '/user',
        type: 'broadcast',
        ...data
      }
    };

    if (actions && Array.isArray(actions)) {
      notificationPayload.actions = actions;
    }

    // Send notifications in batches to avoid overload
    const batchSize = 100;
    const results = [];
    
    for (let i = 0; i < subscriptions.length; i += batchSize) {
      const batch = subscriptions.slice(i, i + batchSize);
      
      const batchPromises = batch.map(async (sub) => {
        try {
          const subscription = JSON.parse(sub.subscription_data);
          
          const result = await webpush.sendNotification(
            subscription,
            JSON.stringify(notificationPayload)
          );
          
          return {
            userId: sub.user_id,
            endpoint: subscription.endpoint,
            success: true,
            statusCode: result.statusCode
          };
        } catch (error) {
          console.error('Error sending notification:', error);
          
          // If subscription is invalid, deactivate
          if (error.statusCode === 410) {
            await connection.execute(
              'UPDATE push_subscriptions SET active = FALSE WHERE subscription_data = ?',
              [sub.subscription_data]
            );
          }
          
          return {
            userId: sub.user_id,
            endpoint: subscription.endpoint,
            success: false,
            error: error.message
          };
        }
      });
      
      const batchResults = await Promise.all(batchPromises);
      results.push(...batchResults);
      
      // Small pause between batches
      if (i + batchSize < subscriptions.length) {
        await new Promise(resolve => setTimeout(resolve, 100));
      }
    }

    // Save broadcast notification log
    await connection.execute(`
      INSERT INTO notification_logs (user_id, title, body, payload, results, sent_at, is_broadcast)
      VALUES (NULL, ?, ?, ?, ?, CURRENT_TIMESTAMP, TRUE)
    `, [
      title,
      body,
      JSON.stringify(notificationPayload),
      JSON.stringify(results)
    ]);

    const successCount = results.filter(r => r.success).length;
    const uniqueUsers = new Set(results.map(r => r.userId)).size;
    
    res.json({
      success: true,
      message: `Notification sent to ${successCount}/${results.length} devices (${uniqueUsers} users)`,
      stats: {
        totalSent: results.length,
        successful: successCount,
        failed: results.length - successCount,
        uniqueUsers
      }
    });
  } catch (error) {
    console.error('Error sending broadcast:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== NOTIFICATION TEMPLATES =====

// New message notification
router.post('/message', authenticateToken, async (req, res) => {
  try {
    const { userId, senderName, message, chatId } = req.body;
    
    const notificationData = {
      userId,
      title: `New message from ${senderName}`,
      body: message.length > 100 ? message.substring(0, 100) + '...' : message,
      tag: `message-${chatId}`,
      data: {
        type: 'message',
        chatId,
        sender: senderName,
        url: `/user?chat=${chatId}`
      },
      actions: [
        {
          action: 'reply',
          title: 'Reply',
          icon: '/icons/reply-icon.png'
        },
        {
          action: 'mark_read',
          title: 'Mark as Read',
          icon: '/icons/read-icon.png'
        }
      ]
    };
    
    // Reuse send endpoint
    req.body = notificationData;
    return router.handle(req, res);
  } catch (error) {
    console.error('Error sending message notification:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Payment notification
router.post('/payment', authenticateToken, async (req, res) => {
  try {
    const { userId, amount, paymentId, status } = req.body;
    
    let title, body;
    
    if (status === 'completed') {
      title = 'Payment Confirmed';
      body = `Payment of R$ ${amount} was confirmed successfully`;
    } else if (status === 'pending') {
      title = 'Payment Pending';
      body = `Payment of R$ ${amount} is being processed`;
    } else {
      title = 'Payment Cancelled';
      body = `Payment of R$ ${amount} was cancelled`;
    }
    
    const notificationData = {
      userId,
      title,
      body,
      tag: `payment-${paymentId}`,
      data: {
        type: 'payment',
        paymentId,
        amount,
        status,
        url: '/payments-admin'
      }
    };
    
    req.body = notificationData;
    return router.handle(req, res);
  } catch (error) {
    console.error('Error sending payment notification:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// ===== STATISTICS =====

// Notification statistics
router.get('/stats', authenticateToken, async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    
    // Active subscriptions
    const [activeSubscriptions] = await connection.execute(
      'SELECT COUNT(*) as count FROM push_subscriptions WHERE active = TRUE'
    );
    
    // Notifications sent today
    const [todayNotifications] = await connection.execute(`
      SELECT COUNT(*) as count 
      FROM notification_logs 
      WHERE DATE(sent_at) = CURDATE()
    `);
    
    // Notifications by type (last 7 days)
    const [typeStats] = await connection.execute(`
      SELECT 
        JSON_EXTRACT(payload, '$.data.type') as type,
        COUNT(*) as count
      FROM notification_logs 
      WHERE sent_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
      GROUP BY JSON_EXTRACT(payload, '$.data.type')
    `);
    
    res.json({
      activeSubscriptions: activeSubscriptions[0].count,
      todayNotifications: todayNotifications[0].count,
      typeStats: typeStats.map(stat => ({
        type: stat.type || 'unknown',
        count: stat.count
      }))
    });
  } catch (error) {
    console.error('Error fetching statistics:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== VAPID PUBLIC KEY =====

// Get VAPID public key
router.get('/vapid-public-key', (req, res) => {
  res.json({ 
    publicKey: vapidKeys.publicKey 
  });
});

module.exports = router;
