const express = require('express');
const crypto = require('crypto');
const { pool } = require('../config/database');
const { authenticateToken, requireAdmin } = require('../middleware/auth');
const PagSeguroService = require('../services/pagseguroService');
const PayPalService = require('../services/paypalService');

const router = express.Router();
router.use(authenticateToken);

// ===== PAYMENT CONFIGURATIONS =====

// List configured payment methods
router.get('/methods', async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const [methods] = await connection.execute(`
      SELECT id, method_name, api_key, api_secret, sandbox_mode, active, created_at, updated_at
      FROM payment_methods 
      ORDER BY method_name ASC
    `);
    
    // Decrypt keys to send to frontend
    const methodsWithKeys = methods.map(method => {
      try {
        return {
          ...method,
          api_key: method.api_key ? decryptData(method.api_key) : null,
          api_secret: method.api_secret ? decryptData(method.api_secret) : null
        };
      } catch (error) {
        console.error('Error decrypting keys:', error);
        return {
          ...method,
          api_key: null,
          api_secret: null
        };
      }
    });
    
    res.json(methodsWithKeys);
  } catch (error) {
    console.error('Error fetching payment methods:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Configure payment method (Admin)
router.post('/methods', requireAdmin, async (req, res) => {
  let connection;
  try {
    const { method_name, api_key, api_secret, sandbox_mode = true } = req.body;
    
    if (!method_name || !api_key) {
      return res.status(400).json({ error: 'Method and API key are required' });
    }

    if (!['paypal', 'pagseguro'].includes(method_name)) {
      return res.status(400).json({ error: 'Payment method not supported' });
    }

    // Encrypt sensitive keys
    const encryptedApiKey = encryptData(api_key);
    const encryptedApiSecret = api_secret ? encryptData(api_secret) : null;

    connection = await pool.getConnection();
    
    // Check if configuration already exists for this method
    const [existing] = await connection.execute(
      'SELECT id FROM payment_methods WHERE method_name = ?',
      [method_name]
    );

    if (existing.length > 0) {
      // Update existing
      await connection.execute(`
        UPDATE payment_methods 
        SET api_key = ?, api_secret = ?, sandbox_mode = ?, updated_at = CURRENT_TIMESTAMP
        WHERE method_name = ?
      `, [encryptedApiKey, encryptedApiSecret, sandbox_mode, method_name]);
    } else {
      // Create new
      await connection.execute(`
        INSERT INTO payment_methods (method_name, api_key, api_secret, sandbox_mode) 
        VALUES (?, ?, ?, ?)
      `, [method_name, encryptedApiKey, encryptedApiSecret, sandbox_mode]);
    }

    res.json({ 
      success: true, 
      message: `${method_name.toUpperCase()} configuration saved successfully` 
    });
  } catch (error) {
    console.error('Error configuring payment method:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Activate/Deactivate payment method (Admin)
router.patch('/methods/:id/toggle', requireAdmin, async (req, res) => {
  let connection;
  try {
    const { id } = req.params;
    const { active } = req.body;

    connection = await pool.getConnection();
    await connection.execute(
      'UPDATE payment_methods SET active = ? WHERE id = ?',
      [active, id]
    );

    res.json({ 
      success: true, 
      message: `Method ${active ? 'activated' : 'deactivated'} successfully` 
    });
  } catch (error) {
    console.error('Error changing method status:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== PAYMENT LINK GENERATION =====

// Create payment link
router.post('/create-link', async (req, res) => {
  let connection;
  try {
    const { 
      payment_method, 
      amount, 
      description, 
      customer_phone, 
      customer_name 
    } = req.body;
    const userId = req.user.userId;

    if (!payment_method || !amount || !customer_phone) {
      return res.status(400).json({ 
        error: 'Payment method, amount and customer phone are required' 
      });
    }

    if (!['paypal', 'pagseguro'].includes(payment_method)) {
      return res.status(400).json({ error: 'Payment method not supported' });
    }

    if (amount <= 0) {
      return res.status(400).json({ error: 'Amount must be greater than zero' });
    }

    connection = await pool.getConnection();
    
    // Check if method is active
    const [method] = await connection.execute(
      'SELECT * FROM payment_methods WHERE method_name = ? AND active = TRUE',
      [payment_method]
    );

    if (method.length === 0) {
      return res.status(400).json({ 
        error: `${payment_method.toUpperCase()} method is not configured or active` 
      });
    }

    const methodConfig = method[0];
    
    // Decrypt keys
    const apiKey = decryptData(methodConfig.api_key);
    const apiSecret = methodConfig.api_secret ? decryptData(methodConfig.api_secret) : null;

    // Generate payment link based on method
    let paymentResult;
    
    if (payment_method === 'paypal') {
      paymentResult = await createPayPalPayment({
        amount,
        description,
        customer_name,
        customer_phone,
        apiKey,
        apiSecret,
        sandbox: methodConfig.sandbox_mode
      });
    } else if (payment_method === 'pagseguro') {
      paymentResult = await createPagSeguroPayment({
        amount,
        description,
        customer_name,
        customer_phone,
        apiKey,
        apiSecret,
        sandbox: methodConfig.sandbox_mode
      });
    }

    if (!paymentResult.success) {
      return res.status(400).json({ error: paymentResult.error });
    }

    // Save to database
    const expiresAt = new Date();
    expiresAt.setHours(expiresAt.getHours() + 24); // Expires in 24 hours

    const [result] = await connection.execute(`
      INSERT INTO payment_links (
        payment_method, amount, description, customer_phone, customer_name,
        payment_url, payment_id, created_by, expires_at
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
    `, [
      payment_method, amount, description, customer_phone, customer_name,
      paymentResult.payment_url, paymentResult.payment_id, userId, expiresAt
    ]);

    res.json({
      success: true,
      payment_link: {
        id: result.insertId,
        payment_url: paymentResult.payment_url,
        payment_id: paymentResult.payment_id,
        amount: amount,
        expires_at: expiresAt
      }
    });
  } catch (error) {
    console.error('Error creating payment link:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// List payment links
router.get('/links', async (req, res) => {
  let connection;
  try {
    const { page = 1, limit = 20, status, method } = req.query;
    const offset = (page - 1) * limit;

    connection = await pool.getConnection();
    
    let query = `
      SELECT p.*, u.username as created_by_name
      FROM payment_links p
      LEFT JOIN users u ON p.created_by = u.id
      WHERE 1=1
    `;
    let params = [];

    if (status) {
      query += ' AND p.status = ?';
      params.push(status);
    }

    if (method) {
      query += ' AND p.payment_method = ?';
      params.push(method);
    }

    query += ' ORDER BY p.created_at DESC LIMIT ? OFFSET ?';
    params.push(parseInt(limit), parseInt(offset));

    const [links] = await connection.execute(query, params);

    // Count total
    let countQuery = 'SELECT COUNT(*) as total FROM payment_links WHERE 1=1';
    let countParams = [];

    if (status) {
      countQuery += ' AND status = ?';
      countParams.push(status);
    }

    if (method) {
      countQuery += ' AND payment_method = ?';
      countParams.push(method);
    }

    const [countResult] = await connection.execute(countQuery, countParams);

    res.json({
      links,
      total: countResult[0].total,
      page: parseInt(page),
      limit: parseInt(limit),
      totalPages: Math.ceil(countResult[0].total / limit)
    });
  } catch (error) {
    console.error('Error fetching payment links:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Check payment status
router.get('/links/:id/status', async (req, res) => {
  let connection;
  try {
    const { id } = req.params;

    connection = await pool.getConnection();
    const [links] = await connection.execute(
      'SELECT * FROM payment_links WHERE id = ?',
      [id]
    );

    if (links.length === 0) {
      return res.status(404).json({ error: 'Payment link not found' });
    }

    const link = links[0];
    
    // Fetch payment method configuration
    const [method] = await connection.execute(
      'SELECT * FROM payment_methods WHERE method_name = ? AND active = TRUE',
      [link.payment_method]
    );

    if (method.length === 0) {
      return res.status(400).json({ 
        error: `${link.payment_method.toUpperCase()} method is not configured or active` 
      });
    }

    const apiKey = decryptData(method[0].api_key);
    const apiSecret = method[0].api_secret ? decryptData(method[0].api_secret) : null;
    const sandbox = method[0].sandbox_mode;
    
    // Check status with payment provider
    let statusResult;
    
    if (link.payment_method === 'paypal') {
      statusResult = await checkPayPalPaymentStatus(link.payment_id, apiKey, apiSecret, sandbox);
    } else if (link.payment_method === 'pagseguro') {
      // PagSeguro: apiKey = token, apiSecret = email
      statusResult = await checkPagSeguroPaymentStatus(link.payment_id, apiSecret, apiKey, sandbox);
    }

    // Update status in database if changed
    if (statusResult && statusResult.status !== link.status) {
      await connection.execute(
        'UPDATE payment_links SET status = ?, paid_at = ? WHERE id = ?',
        [statusResult.status, statusResult.paid_at, id]
      );
      
      link.status = statusResult.status;
      link.paid_at = statusResult.paid_at;
    }

    res.json({
      id: link.id,
      status: link.status,
      amount: link.amount,
      paid_at: link.paid_at,
      expires_at: link.expires_at
    });
  } catch (error) {
    console.error('Error checking payment status:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Webhook for PayPal
router.post('/webhook/paypal', async (req, res) => {
  let connection;
  try {
    const event = req.body;
    
    // Verify webhook signature (implement real validation)
    if (!validatePayPalWebhook(event, req.headers)) {
      return res.status(401).json({ error: 'Invalid webhook' });
    }

    connection = await pool.getConnection();

    // Process different event types
    if (event.event_type === 'PAYMENT.CAPTURE.COMPLETED' || event.event_type === 'CHECKOUT.ORDER.APPROVED') {
      const orderId = event.resource.id || event.resource.order_id;
      
      await connection.execute(`
        UPDATE payment_links 
        SET status = 'paid', paid_at = CURRENT_TIMESTAMP 
        WHERE payment_id = ? AND status = 'pending'
      `, [orderId]);
    } else if (event.event_type === 'CHECKOUT.ORDER.CANCELLED') {
      const orderId = event.resource.id;
      
      await connection.execute(`
        UPDATE payment_links 
        SET status = 'cancelled' 
        WHERE payment_id = ? AND status = 'pending'
      `, [orderId]);
    }

    res.json({ success: true });
  } catch (error) {
    console.error('Error in PayPal webhook:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Webhook for PagSeguro
router.post('/webhook/pagseguro', async (req, res) => {
  let connection;
  try {
    const notification = req.body;
    
    // Verify PagSeguro notification
    if (notification.notificationType === 'transaction') {
      const transactionCode = notification.notificationCode;
      
      connection = await pool.getConnection();
      
      // Fetch PagSeguro configuration
      const [method] = await connection.execute(
        'SELECT * FROM payment_methods WHERE method_name = ? AND active = TRUE',
        ['pagseguro']
      );
      
      if (method.length > 0) {
        const apiKey = decryptData(method[0].api_key);
        const apiSecret = method[0].api_secret ? decryptData(method[0].api_secret) : null;
        const sandbox = method[0].sandbox_mode;
        
        // Query transaction status in PagSeguro
        // PagSeguro: apiKey = token, apiSecret = email
        const statusResult = await checkPagSeguroTransaction(transactionCode, apiSecret, apiKey, sandbox);
        
        if (statusResult.success && statusResult.reference) {
          await connection.execute(`
            UPDATE payment_links 
            SET status = ?, paid_at = ? 
            WHERE payment_id = ?
          `, [statusResult.status, statusResult.paid_at, statusResult.reference]);
        }
      }
    }

    res.json({ success: true });
  } catch (error) {
    console.error('Error in PagSeguro webhook:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== HELPER FUNCTIONS =====

// Encryption for sensitive data
function encryptData(text) {
  const algorithm = 'aes-256-cbc';
  const keyString = process.env.ENCRYPTION_KEY || 'default-key-change-in-production-32';
  // Create 32-byte key (256 bits)
  const key = crypto.createHash('sha256').update(keyString).digest();
  const iv = crypto.randomBytes(16);
  
  const cipher = crypto.createCipheriv(algorithm, key, iv);
  let encrypted = cipher.update(text, 'utf8', 'hex');
  encrypted += cipher.final('hex');
  
  return iv.toString('hex') + ':' + encrypted;
}

function decryptData(encryptedText) {
  const algorithm = 'aes-256-cbc';
  const keyString = process.env.ENCRYPTION_KEY || 'default-key-change-in-production-32';
  // Create 32-byte key (256 bits)
  const key = crypto.createHash('sha256').update(keyString).digest();
  
  const textParts = encryptedText.split(':');
  const iv = Buffer.from(textParts.shift(), 'hex');
  const encrypted = textParts.join(':');
  
  const decipher = crypto.createDecipheriv(algorithm, key, iv);
  let decrypted = decipher.update(encrypted, 'hex', 'utf8');
  decrypted += decipher.final('utf8');
  
  return decrypted;
}

// PayPal functions - Real implementation with SDK
async function createPayPalPayment(data) {
  try {
    const paypalService = new PayPalService(
      data.apiKey,
      data.apiSecret,
      data.sandbox
    );
    
    const result = await paypalService.createPayment({
      amount: data.amount,
      description: data.description,
      customer_name: data.customer_name,
      customer_phone: data.customer_phone,
      reference_id: 'REF_' + Date.now()
    });

    return result;
  } catch (error) {
    return {
      success: false,
      error: 'Error creating PayPal payment: ' + error.message
    };
  }
}

async function checkPayPalPaymentStatus(paymentId, clientId, clientSecret, sandbox) {
  try {
    const paypalService = new PayPalService(
      clientId,
      clientSecret,
      sandbox
    );
    
    const result = await paypalService.getPaymentStatus(paymentId);
    
    return {
      status: result.status || 'pending',
      paid_at: result.paid_at || null
    };
  } catch (error) {
    console.error('Error checking PayPal status:', error);
    return {
      status: 'pending',
      paid_at: null
    };
  }
}

function validatePayPalWebhook(event, headers) {
  // Implement real PayPal webhook validation
  return true;
}

// PagSeguro functions - Real implementation with API
async function createPagSeguroPayment(data) {
  try {
    // PagSeguro: apiKey = email, apiSecret = token
    const pagseguroService = new PagSeguroService(
      data.apiKey,    // email
      data.apiSecret, // token
      data.sandbox
    );
    
    const result = await pagseguroService.createPayment({
      amount: data.amount,
      description: data.description,
      customer_name: data.customer_name,
      customer_phone: data.customer_phone,
      reference_id: 'REF_' + Date.now()
    });

    return result;
  } catch (error) {
    return {
      success: false,
      error: 'Error creating PagSeguro payment: ' + error.message
    };
  }
}

async function checkPagSeguroPaymentStatus(paymentId, email, token, sandbox) {
  try {
    const pagseguroService = new PagSeguroService(
      email,
      token,
      sandbox
    );
    
    const result = await pagseguroService.getPaymentStatus(paymentId);
    
    return {
      status: result.status || 'pending',
      paid_at: result.paid_at || null
    };
  } catch (error) {
    console.error('Error checking PagSeguro status:', error);
    return {
      status: 'pending',
      paid_at: null
    };
  }
}

async function checkPagSeguroTransaction(notificationCode, email, token, sandbox) {
  try {
    const pagseguroService = new PagSeguroService(
      email,
      token,
      sandbox
    );
    
    const result = await pagseguroService.getTransactionByNotification(notificationCode);
    
    if (result.success && result.transaction) {
      const statusInfo = pagseguroService.parseTransactionStatus(
        `<status>${result.transaction.status}</status><lastEventDate>${result.transaction.date}</lastEventDate>`
      );
      
      return {
        success: true,
        status: statusInfo.status,
        paid_at: statusInfo.paid_at,
        reference: result.transaction.reference
      };
    }
    
    return {
      success: false,
      status: 'pending',
      paid_at: null,
      reference: null
    };
  } catch (error) {
    console.error('Error checking PagSeguro transaction:', error);
    return {
      success: false,
      status: 'pending',
      paid_at: null,
      reference: null
    };
  }
}

module.exports = router;
