const express = require('express');
const multer = require('multer');
const csv = require('csv-parser');
const fs = require('fs');
const { pool } = require('../config/database');
const { authenticateToken } = require('../middleware/auth');

const router = express.Router();
router.use(authenticateToken);

// Configure multer for file upload
const upload = multer({ dest: 'uploads/' });

// List contacts
router.get('/contacts', async (req, res) => {
  try {
    const connection = await pool.getConnection();
    const [contacts] = await connection.execute(
      'SELECT * FROM contacts ORDER BY created_at DESC'
    );
    connection.release();

    res.json(contacts);
  } catch (error) {
    console.error('Error fetching contacts:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Add contact
router.post('/contacts', async (req, res) => {
  try {
    const { phone_number, name } = req.body;

    if (!phone_number) {
      return res.status(400).json({ error: 'Phone number is required' });
    }

    const connection = await pool.getConnection();
    const [result] = await connection.execute(
      'INSERT INTO contacts (phone_number, name) VALUES (?, ?) ON DUPLICATE KEY UPDATE name = ?',
      [phone_number, name, name]
    );
    connection.release();

    res.json({ 
      message: 'Contact added successfully',
      contactId: result.insertId 
    });
  } catch (error) {
    console.error('Error adding contact:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Import contacts via CSV
router.post('/contacts/import', upload.single('csvFile'), async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ error: 'CSV file is required' });
    }

    const contacts = [];
    const connection = await pool.getConnection();

    fs.createReadStream(req.file.path)
      .pipe(csv())
      .on('data', (row) => {
        if (row.phone_number) {
          contacts.push([row.phone_number, row.name || '']);
        }
      })
      .on('end', async () => {
        try {
          // Insert contacts in batch
          if (contacts.length > 0) {
            const placeholders = contacts.map(() => '(?, ?)').join(',');
            const values = contacts.flat();
            
            await connection.execute(
              `INSERT INTO contacts (phone_number, name) VALUES ${placeholders} ON DUPLICATE KEY UPDATE name = VALUES(name)`,
              values
            );
          }

          connection.release();
          
          // Remove temporary file
          fs.unlinkSync(req.file.path);

          res.json({ 
            message: `${contacts.length} contacts imported successfully` 
          });
        } catch (error) {
          connection.release();
          console.error('Error importing contacts:', error);
          res.status(500).json({ error: 'Error processing CSV file' });
        }
      });

  } catch (error) {
    console.error('Error in CSV upload:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Export contacts to CSV
router.get('/contacts/export', async (req, res) => {
  try {
    const connection = await pool.getConnection();
    const [contacts] = await connection.execute(
      'SELECT phone_number, name FROM contacts ORDER BY created_at DESC'
    );
    connection.release();

    // Generate CSV
    let csvContent = 'phone_number,name\n';
    contacts.forEach(contact => {
      csvContent += `${contact.phone_number},"${contact.name || ''}"\n`;
    });

    res.setHeader('Content-Type', 'text/csv');
    res.setHeader('Content-Disposition', 'attachment; filename=contacts.csv');
    res.send(csvContent);

  } catch (error) {
    console.error('Error exporting contacts:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Create mass sending campaign
router.post('/campaigns', async (req, res) => {
  try {
    const { message_text, message_type = 'text', media_url, contact_ids } = req.body;
    const { userId } = req.user;

    if (!message_text || !contact_ids || contact_ids.length === 0) {
      return res.status(400).json({ error: 'Message and contacts are required' });
    }

    const connection = await pool.getConnection();
    
    // Create campaign
    const [result] = await connection.execute(
      'INSERT INTO mass_messages (message_text, message_type, media_url, total_contacts, created_by) VALUES (?, ?, ?, ?, ?)',
      [message_text, message_type, media_url, contact_ids.length, userId]
    );

    const campaignId = result.insertId;

    // Fetch contacts
    const placeholders = contact_ids.map(() => '?').join(',');
    const [contacts] = await connection.execute(
      `SELECT phone_number FROM contacts WHERE id IN (${placeholders})`,
      contact_ids
    );

    connection.release();

    // Start mass sending (implement in background)
    startMassSending(campaignId, contacts, message_text, message_type, media_url);

    res.json({ 
      message: 'Campaign created successfully',
      campaignId: campaignId 
    });

  } catch (error) {
    console.error('Error creating campaign:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Function for mass sending (background)
async function startMassSending(campaignId, contacts, messageText, messageType, mediaUrl) {
  try {
    const connection = await pool.getConnection();
    
    // Update status to "sending"
    await connection.execute(
      'UPDATE mass_messages SET status = ? WHERE id = ?',
      ['sending', campaignId]
    );

    let sentCount = 0;

    for (const contact of contacts) {
      try {
        // Send message via WhatsApp
        // Note: Implement integration with whatsappService here
        console.log(`Sending to ${contact.phone_number}: ${messageText}`);
        
        sentCount++;
        
        // Update counter
        await connection.execute(
          'UPDATE mass_messages SET sent_count = ? WHERE id = ?',
          [sentCount, campaignId]
        );

        // Wait 21 seconds between sends
        await new Promise(resolve => setTimeout(resolve, 21000));

      } catch (error) {
        console.error(`Error sending to ${contact.phone_number}:`, error);
      }
    }

    // Mark as completed
    await connection.execute(
      'UPDATE mass_messages SET status = ? WHERE id = ?',
      ['completed', campaignId]
    );

    connection.release();

  } catch (error) {
    console.error('Error in mass sending:', error);
    
    // Mark as failed
    const connection = await pool.getConnection();
    await connection.execute(
      'UPDATE mass_messages SET status = ? WHERE id = ?',
      ['failed', campaignId]
    );
    connection.release();
  }
}

// List campaigns
router.get('/campaigns', async (req, res) => {
  try {
    const connection = await pool.getConnection();
    const [campaigns] = await connection.execute(`
      SELECT mm.*, u.username as created_by_username
      FROM mass_messages mm
      LEFT JOIN users u ON mm.created_by = u.id
      ORDER BY mm.created_at DESC
    `);
    connection.release();

    res.json(campaigns);
  } catch (error) {
    console.error('Error fetching campaigns:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

module.exports = router;

