const express = require('express');
const multer = require('multer');
const csv = require('csv-parser');
const fs = require('fs');
const path = require('path');
const { pool } = require('../config/database');
const { authenticateToken, requireAdmin } = require('../middleware/auth');

const router = express.Router();
router.use(authenticateToken);

// Configure multer for file upload
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    const uploadDir = 'uploads/';
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }
    cb(null, uploadDir);
  },
  filename: function (req, file, cb) {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, file.fieldname + '-' + uniqueSuffix + path.extname(file.originalname));
  }
});

const upload = multer({ 
  storage: storage,
  limits: {
    fileSize: 50 * 1024 * 1024 // 50MB
  },
  fileFilter: function (req, file, cb) {
    // Allow CSV for contacts and media for messages
    const allowedTypes = [
      'text/csv',
      'application/vnd.ms-excel',
      'image/jpeg',
      'image/png',
      'image/gif',
      'video/mp4',
      'video/avi',
      'audio/mpeg',
      'audio/wav',
      'application/pdf',
      'application/msword',
      'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
    ];
    
    if (allowedTypes.includes(file.mimetype)) {
      cb(null, true);
    } else {
      cb(new Error('File type not allowed'), false);
    }
  }
});

// ===== CONTACT GROUPS =====

// List contact groups
router.get('/groups', async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const [groups] = await connection.execute(`
      SELECT g.*, 
             COUNT(c.id) as contact_count,
             u.username as created_by_name
      FROM contact_groups g
      LEFT JOIN contacts c ON g.id = c.group_id
      LEFT JOIN users u ON g.created_by = u.id
      GROUP BY g.id
      ORDER BY g.created_at DESC
    `);
    res.json(groups);
  } catch (error) {
    console.error('Error fetching groups:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Create contact group
router.post('/groups', async (req, res) => {
  let connection;
  try {
    const { group_name, description } = req.body;
    const userId = req.user.userId;

    if (!group_name) {
      return res.status(400).json({ error: 'Group name is required' });
    }

    connection = await pool.getConnection();
    const [result] = await connection.execute(
      'INSERT INTO contact_groups (group_name, description, created_by) VALUES (?, ?, ?)',
      [group_name, description, userId]
    );

    res.json({ 
      success: true,
      message: 'Group created successfully',
      groupId: result.insertId 
    });
  } catch (error) {
    console.error('Error creating group:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Update group
router.put('/groups/:id', async (req, res) => {
  let connection;
  try {
    const { id } = req.params;
    const { group_name, description } = req.body;

    connection = await pool.getConnection();
    await connection.execute(
      'UPDATE contact_groups SET group_name = ?, description = ? WHERE id = ?',
      [group_name, description, id]
    );

    res.json({ success: true, message: 'Group updated successfully' });
  } catch (error) {
    console.error('Error updating group:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Delete group
router.delete('/groups/:id', async (req, res) => {
  let connection;
  try {
    const { id } = req.params;

    connection = await pool.getConnection();
    await connection.beginTransaction();

    // Move contacts to default group
    const [defaultGroup] = await connection.execute(
      'SELECT id FROM contact_groups WHERE group_name = ? LIMIT 1',
      ['Clientes Gerais']
    );

    if (defaultGroup.length > 0) {
      await connection.execute(
        'UPDATE contacts SET group_id = ? WHERE group_id = ?',
        [defaultGroup[0].id, id]
      );
    }

    // Delete group
    await connection.execute('DELETE FROM contact_groups WHERE id = ?', [id]);
    
    await connection.commit();
    res.json({ success: true, message: 'Group removed successfully' });
  } catch (error) {
    if (connection) await connection.rollback();
    console.error('Error removing group:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== CONTACTS =====

// List contacts with filters
router.get('/contacts', async (req, res) => {
  let connection;
  try {
    const { group_id, search, page = 1, limit = 50 } = req.query;
    const offset = (page - 1) * limit;

    connection = await pool.getConnection();
    
    let query = `
      SELECT c.*, g.group_name 
      FROM contacts c
      LEFT JOIN contact_groups g ON c.group_id = g.id
      WHERE 1=1
    `;
    let params = [];

    if (group_id) {
      query += ' AND c.group_id = ?';
      params.push(group_id);
    }

    if (search) {
      query += ' AND (c.name LIKE ? OR c.phone_number LIKE ?)';
      params.push(`%${search}%`, `%${search}%`);
    }

    query += ' ORDER BY c.created_at DESC LIMIT ? OFFSET ?';
    params.push(parseInt(limit), parseInt(offset));

    const [contacts] = await connection.execute(query, params);

    // Count total
    let countQuery = 'SELECT COUNT(*) as total FROM contacts c WHERE 1=1';
    let countParams = [];

    if (group_id) {
      countQuery += ' AND c.group_id = ?';
      countParams.push(group_id);
    }

    if (search) {
      countQuery += ' AND (c.name LIKE ? OR c.phone_number LIKE ?)';
      countParams.push(`%${search}%`, `%${search}%`);
    }

    const [countResult] = await connection.execute(countQuery, countParams);

    res.json({
      contacts,
      total: countResult[0].total,
      page: parseInt(page),
      limit: parseInt(limit),
      totalPages: Math.ceil(countResult[0].total / limit)
    });
  } catch (error) {
    console.error('Error fetching contacts:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Add individual contact
router.post('/contacts', async (req, res) => {
  let connection;
  try {
    const { phone_number, name, group_id, tags, notes } = req.body;

    if (!phone_number) {
      return res.status(400).json({ error: 'Phone number is required' });
    }

    // Clean and validate number
    const cleanPhone = phone_number.replace(/\D/g, '');
    if (cleanPhone.length < 10) {
      return res.status(400).json({ error: 'Invalid phone number' });
    }

    connection = await pool.getConnection();
    
    // Check if group exists
    let finalGroupId = group_id;
    if (group_id) {
      const [groupExists] = await connection.execute(
        'SELECT id FROM contact_groups WHERE id = ?',
        [group_id]
      );
      if (groupExists.length === 0) {
        finalGroupId = null;
      }
    }

    const [result] = await connection.execute(`
      INSERT INTO contacts (phone_number, name, group_id, tags, notes) 
      VALUES (?, ?, ?, ?, ?) 
      ON DUPLICATE KEY UPDATE 
        name = VALUES(name),
        group_id = VALUES(group_id),
        tags = VALUES(tags),
        notes = VALUES(notes)
    `, [cleanPhone, name, finalGroupId, tags, notes]);

    res.json({ 
      success: true,
      message: 'Contact added successfully',
      contactId: result.insertId || result.affectedRows
    });
  } catch (error) {
    console.error('Error adding contact:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Save contact from conversation
router.post('/contacts/from-conversation', async (req, res) => {
  let connection;
  try {
    const { phone_number, name, group_id } = req.body;

    if (!phone_number) {
      return res.status(400).json({ error: 'Phone number is required' });
    }

    connection = await pool.getConnection();
    
    // Fetch conversation name if not provided
    let contactName = name;
    if (!contactName) {
      const [conversation] = await connection.execute(
        'SELECT contact_name FROM conversations WHERE phone_number = ? LIMIT 1',
        [phone_number]
      );
      if (conversation.length > 0) {
        contactName = conversation[0].contact_name;
      }
    }

    const [result] = await connection.execute(`
      INSERT INTO contacts (phone_number, name, group_id) 
      VALUES (?, ?, ?) 
      ON DUPLICATE KEY UPDATE 
        name = COALESCE(VALUES(name), name),
        group_id = COALESCE(VALUES(group_id), group_id)
    `, [phone_number, contactName, group_id]);

    res.json({ 
      success: true,
      message: 'Contact saved from conversation',
      contactId: result.insertId || result.affectedRows
    });
  } catch (error) {
    console.error('Error saving contact from conversation:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Import contacts via CSV
router.post('/contacts/import', upload.single('csvFile'), async (req, res) => {
  let connection;
  try {
    if (!req.file) {
      return res.status(400).json({ error: 'CSV file is required' });
    }

    const { group_id } = req.body;
    const filePath = req.file.path;
    const contacts = [];
    let errors = [];

    // Read CSV file
    await new Promise((resolve, reject) => {
      fs.createReadStream(filePath)
        .pipe(csv())
        .on('data', (row) => {
          // Accept different header formats
          const phone = row.phone || row.telefone || row.numero || row.phone_number;
          const name = row.name || row.nome || row.contact_name;
          const tags = row.tags || row.etiquetas || '';
          const notes = row.notes || row.observacoes || row.obs || '';

          if (phone) {
            const cleanPhone = phone.toString().replace(/\D/g, '');
            if (cleanPhone.length >= 10) {
              contacts.push({
                phone_number: cleanPhone,
                name: name || '',
                tags: tags,
                notes: notes
              });
            } else {
              errors.push(`Invalid number: ${phone}`);
            }
          }
        })
        .on('end', resolve)
        .on('error', reject);
    });

    // Insert contacts into database
    connection = await pool.getConnection();
    await connection.beginTransaction();

    let imported = 0;
    let updated = 0;

    for (const contact of contacts) {
      try {
        const [result] = await connection.execute(`
          INSERT INTO contacts (phone_number, name, group_id, tags, notes) 
          VALUES (?, ?, ?, ?, ?) 
          ON DUPLICATE KEY UPDATE 
            name = COALESCE(VALUES(name), name),
            group_id = COALESCE(VALUES(group_id), group_id),
            tags = COALESCE(VALUES(tags), tags),
            notes = COALESCE(VALUES(notes), notes)
        `, [contact.phone_number, contact.name, group_id, contact.tags, contact.notes]);

        if (result.insertId) {
          imported++;
        } else {
          updated++;
        }
      } catch (error) {
        errors.push(`Error importing ${contact.phone_number}: ${error.message}`);
      }
    }

    await connection.commit();

    // Clean temporary file
    fs.unlinkSync(filePath);

    res.json({
      success: true,
      message: 'Import completed',
      imported,
      updated,
      errors: errors.slice(0, 10), // Limit errors shown
      totalErrors: errors.length
    });
  } catch (error) {
    if (connection) await connection.rollback();
    console.error('Error importing contacts:', error);
    
    // Clean file in case of error
    if (req.file && fs.existsSync(req.file.path)) {
      fs.unlinkSync(req.file.path);
    }
    
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// ===== MASS MESSAGES =====

// List mass sending campaigns
router.get('/campaigns', async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    
    // Check if mass_messages table exists
    const [tables] = await connection.execute(
      "SHOW TABLES LIKE 'mass_messages'"
    );
    
    if (tables.length === 0) {
      // Table doesn't exist, return empty array
      return res.json([]);
    }
    
    const [campaigns] = await connection.execute(`
      SELECT m.*, 
             g.group_name,
             u.username as created_by_name,
             COUNT(l.id) as total_logs,
             SUM(CASE WHEN l.status = 'sent' THEN 1 ELSE 0 END) as sent_count,
             SUM(CASE WHEN l.status = 'failed' THEN 1 ELSE 0 END) as failed_count
      FROM mass_messages m
      LEFT JOIN contact_groups g ON m.group_id = g.id
      LEFT JOIN users u ON m.created_by = u.id
      LEFT JOIN mass_message_logs l ON m.id = l.mass_message_id
      GROUP BY m.id
      ORDER BY m.created_at DESC
    `);
    res.json(campaigns);
  } catch (error) {
    console.error('Error fetching campaigns:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Create mass sending campaign
router.post('/campaigns', upload.single('mediaFile'), async (req, res) => {
  let connection;
  try {
    const { 
      message_text, 
      message_type = 'text',
      group_id,
      send_interval = 5,
      scheduled_at
    } = req.body;
    const userId = req.user.userId;

    if (!message_text && !req.file) {
      return res.status(400).json({ error: 'Message text or file is required' });
    }

    let media_url = null;
    let finalMessageType = message_type;

    // Process media file
    if (req.file) {
      media_url = `/uploads/${req.file.filename}`;
      
      // Determine type based on file
      if (req.file.mimetype.startsWith('image/')) {
        finalMessageType = 'image';
      } else if (req.file.mimetype.startsWith('video/')) {
        finalMessageType = 'video';
      } else if (req.file.mimetype.startsWith('audio/')) {
        finalMessageType = 'audio';
      } else {
        finalMessageType = 'document';
      }
    }

    connection = await pool.getConnection();
    
    // Count group contacts
    let contactCount = 0;
    if (group_id) {
      const [count] = await connection.execute(
        'SELECT COUNT(*) as total FROM contacts WHERE group_id = ?',
        [group_id]
      );
      contactCount = count[0].total;
    } else {
      const [count] = await connection.execute('SELECT COUNT(*) as total FROM contacts');
      contactCount = count[0].total;
    }

    // Create campaign
    const [result] = await connection.execute(`
      INSERT INTO mass_messages (
        message_text, media_url, message_type, total_contacts, 
        group_id, send_interval, scheduled_at, created_by, status
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'pending')
    `, [
      message_text, media_url, finalMessageType, contactCount,
      group_id, send_interval, scheduled_at, userId
    ]);

    res.json({
      success: true,
      message: 'Campaign created successfully',
      campaignId: result.insertId,
      contactCount
    });
  } catch (error) {
    console.error('Error creating campaign:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Start campaign sending
router.post('/campaigns/:id/start', async (req, res) => {
  let connection;
  try {
    const { id } = req.params;

    connection = await pool.getConnection();
    await connection.beginTransaction();

    // Fetch campaign
    const [campaign] = await connection.execute(
      'SELECT * FROM mass_messages WHERE id = ? AND status = "pending"',
      [id]
    );

    if (campaign.length === 0) {
      return res.status(404).json({ error: 'Campaign not found or already started' });
    }

    const campaignData = campaign[0];

    // Fetch contacts
    let contactQuery = 'SELECT phone_number, name FROM contacts';
    let contactParams = [];

    if (campaignData.group_id) {
      contactQuery += ' WHERE group_id = ?';
      contactParams.push(campaignData.group_id);
    }

    const [contacts] = await connection.execute(contactQuery, contactParams);

    // Create logs for each contact
    for (const contact of contacts) {
      await connection.execute(`
        INSERT INTO mass_message_logs (
          mass_message_id, contact_phone, contact_name, status
        ) VALUES (?, ?, ?, 'pending')
      `, [id, contact.phone_number, contact.name]);
    }

    // Update campaign status
    await connection.execute(
      'UPDATE mass_messages SET status = "sending" WHERE id = ?',
      [id]
    );

    await connection.commit();

    // Here you would integrate with WhatsApp Service for real sending
    // For now, we'll simulate the process

    res.json({
      success: true,
      message: 'Sending started',
      contactCount: contacts.length
    });
  } catch (error) {
    if (connection) await connection.rollback();
    console.error('Error starting sending:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Sending logs for a campaign
router.get('/campaigns/:id/logs', async (req, res) => {
  let connection;
  try {
    const { id } = req.params;
    const { page = 1, limit = 50, status } = req.query;
    const offset = (page - 1) * limit;

    connection = await pool.getConnection();
    
    let query = `
      SELECT * FROM mass_message_logs 
      WHERE mass_message_id = ?
    `;
    let params = [id];

    if (status) {
      query += ' AND status = ?';
      params.push(status);
    }

    query += ' ORDER BY created_at DESC LIMIT ? OFFSET ?';
    params.push(parseInt(limit), parseInt(offset));

    const [logs] = await connection.execute(query, params);

    // Count total
    let countQuery = 'SELECT COUNT(*) as total FROM mass_message_logs WHERE mass_message_id = ?';
    let countParams = [id];

    if (status) {
      countQuery += ' AND status = ?';
      countParams.push(status);
    }

    const [countResult] = await connection.execute(countQuery, countParams);

    res.json({
      logs,
      total: countResult[0].total,
      page: parseInt(page),
      limit: parseInt(limit),
      totalPages: Math.ceil(countResult[0].total / limit)
    });
  } catch (error) {
    console.error('Error fetching logs:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Sending statistics
router.get('/campaigns/:id/stats', async (req, res) => {
  let connection;
  try {
    const { id } = req.params;

    connection = await pool.getConnection();
    const [stats] = await connection.execute(`
      SELECT 
        status,
        COUNT(*) as count
      FROM mass_message_logs 
      WHERE mass_message_id = ?
      GROUP BY status
    `, [id]);

    const result = {
      pending: 0,
      sent: 0,
      failed: 0,
      delivered: 0
    };

    stats.forEach(stat => {
      result[stat.status] = stat.count;
    });

    res.json(result);
  } catch (error) {
    console.error('Error fetching statistics:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

module.exports = router;
