/**
 * LICENSE ROUTES FOR CLIENT SYSTEM
 * 
 * This file should be added to the MISAYAN system that will be sold.
 * It manages the local license configuration.
 * 
 * Installation:
 * 1. Copy this file to the /routes/ folder of your project
 * 2. Add in server.js:
 *    app.use('/api/license', require('./routes/license-routes'));
 */

const express = require('express');
const router = express.Router();
const fs = require('fs');
const path = require('path');
const { validateLicenseRemote } = require('../middleware/licenseValidator');

// File where the license will be stored
const LICENSE_FILE = path.join(__dirname, '../.license');

/**
 * POST /api/license/save
 * Saves the license key to the local file
 */
router.post('/save', async (req, res) => {
  try {
    const { license_key } = req.body;
    
    if (!license_key) {
      return res.status(400).json({
        success: false,
        error: 'License key is required'
      });
    }
    
    // Validate basic format
    if (!license_key.startsWith('eyJ')) {
      return res.status(400).json({
        success: false,
        error: 'Invalid license format'
      });
    }
    
    // Validate remotely before saving
    const validation = await validateLicenseRemote(license_key);
    
    if (!validation.valid) {
      return res.status(400).json({
        success: false,
        error: validation.error || 'Invalid license',
        revoked: validation.revoked,
        expired: validation.expired
      });
    }
    
    // Save to file
    fs.writeFileSync(LICENSE_FILE, license_key, 'utf8');
    
    // Also save to .env (optional, but recommended)
    const envPath = path.join(__dirname, '../.env');
    if (fs.existsSync(envPath)) {
      let envContent = fs.readFileSync(envPath, 'utf8');
      
      // Remove old LICENSE_KEY line if it exists
      envContent = envContent.replace(/^LICENSE_KEY=.*/gm, '');
      
      // Add new LICENSE_KEY
      envContent += `\nLICENSE_KEY=${license_key}\n`;
      
      fs.writeFileSync(envPath, envContent, 'utf8');
    }
    
    res.json({
      success: true,
      message: 'License saved successfully',
      buyer: validation.buyer,
      domain: validation.domain,
      expires_at: validation.expires_at
    });
    
  } catch (error) {
    console.error('Error saving license:', error);
    res.status(500).json({
      success: false,
      error: 'Error saving license'
    });
  }
});

/**
 * GET /api/license/status
 * Checks the status of the current license
 */
router.get('/status', async (req, res) => {
  try {
    // Try to read from file
    let licenseKey = null;
    
    if (fs.existsSync(LICENSE_FILE)) {
      licenseKey = fs.readFileSync(LICENSE_FILE, 'utf8').trim();
    }
    
    // If not found in file, try from .env
    if (!licenseKey) {
      licenseKey = process.env.LICENSE_KEY;
    }
    
    if (!licenseKey) {
      return res.json({
        configured: false,
        valid: false,
        error: 'No license configured'
      });
    }
    
    // Validate remotely
    const validation = await validateLicenseRemote(licenseKey);
    
    if (validation.valid) {
      res.json({
        configured: true,
        valid: true,
        buyer: validation.buyer,
        domain: validation.domain,
        created_at: validation.created_at,
        expires_at: validation.expires_at
      });
    } else {
      res.json({
        configured: true,
        valid: false,
        error: validation.error,
        revoked: validation.revoked,
        expired: validation.expired
      });
    }
    
  } catch (error) {
    console.error('Error checking license status:', error);
    res.status(500).json({
      configured: false,
      valid: false,
      error: 'Error checking license'
    });
  }
});

/**
 * DELETE /api/license/remove
 * Removes the configured license (requires admin authentication)
 */
router.delete('/remove', async (req, res) => {
  try {
    // Remove license file
    if (fs.existsSync(LICENSE_FILE)) {
      fs.unlinkSync(LICENSE_FILE);
    }
    
    // Remove from .env
    const envPath = path.join(__dirname, '../.env');
    if (fs.existsSync(envPath)) {
      let envContent = fs.readFileSync(envPath, 'utf8');
      envContent = envContent.replace(/^LICENSE_KEY=.*/gm, '');
      fs.writeFileSync(envPath, envContent, 'utf8');
    }
    
    res.json({
      success: true,
      message: 'License removed successfully'
    });
    
  } catch (error) {
    console.error('Error removing license:', error);
    res.status(500).json({
      success: false,
      error: 'Error removing license'
    });
  }
});

module.exports = router;

