const express = require('express');
const { pool } = require('../config/database');
const { authenticateToken } = require('../middleware/auth');

const router = express.Router();
router.use(authenticateToken);

// ===================================================================
//  NEW ROUTE ADDED HERE
//  Objective: Allow any logged-in user to fetch the list of stores.
// ===================================================================
router.get('/stores', async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    // Select only the ID and name of stores, ordered by name.
    const [stores] = await connection.execute('SELECT id, name FROM stores ORDER BY name');
    connection.release();
    res.json(stores);
  } catch (error) {
    if (connection) connection.release();
    console.error('Error listing stores:', error);
    res.status(500).json({ error: 'Internal server error fetching stores' });
  }
});
// ===================================================================
//  END OF NEW ROUTE
// ===================================================================

// Create new conversation
router.post('/conversations', async (req, res) => {
    let connection;
    try {
        const { phone_number, contact_name } = req.body;
        const userId = req.user.userId || req.user.id;
        const userStore = req.user.store;
        
        if (!phone_number) {
            return res.status(400).json({ error: 'Phone number is required' });
        }
        
        connection = await pool.getConnection();
        
        // Check if a conversation with this number already exists
        const [existing] = await connection.execute(
            'SELECT id FROM conversations WHERE phone_number = ? AND (assigned_store = ? OR assigned_store IS NULL)',
            [phone_number, userStore]
        );
        
        if (existing.length > 0) {
            return res.json({ id: existing[0].id, message: 'Conversation already exists' });
        }
        
        // Create new conversation
        const [result] = await connection.execute(
            `INSERT INTO conversations 
            (phone_number, contact_name, assigned_store, assigned_user_id, status, created_at) 
            VALUES (?, ?, ?, ?, 'waiting', NOW())`,
            [phone_number, contact_name || null, userStore, userId]
        );
        
        res.status(201).json({
            id: result.insertId,
            phone_number,
            contact_name,
            message: 'Conversation created successfully'
        });
        
    } catch (error) {
        console.error('Error creating conversation:', error);
        res.status(500).json({ error: 'Error creating conversation' });
    } finally {
        if (connection) connection.release();
    }
});


// List conversations
router.get('/conversations', async (req, res) => {
  let connection;
  try {
    const { role, store } = req.user;
    
    // Debug logging
    console.log('User data:', { role, store });
    console.log('Full user object:', req.user);
    
    connection = await pool.getConnection();

    let query = `
      SELECT c.*, u.username as assigned_username
      FROM conversations c
      LEFT JOIN users u ON c.assigned_user_id = u.id
      WHERE 1=1
    `;
    let params = [];

    // CORRECTED LOGIC:
    // - STORE users see NULL conversations (available) + conversations from their store
    // - DEPARTMENT users see conversations with assigned_store starting with 'SETOR:' or 'DEPT:'
    if (role !== 'admin') {
      if (store && store !== null && store !== undefined) {
        // Store user: see NULL conversations (new) OR from their store
        query += ' AND (c.assigned_store IS NULL OR c.assigned_store = ?)';
        params.push(store);
      } else {
        // Department user: see conversations marked with department prefix
        query += ' AND (c.assigned_store LIKE "SETOR:%" OR c.assigned_store LIKE "DEPT:%")';
      }
    }

    query += ' ORDER BY c.last_message_time DESC';

    console.log('Executing query:', query);
    console.log('With params:', params);

    const [conversations] = await connection.execute(query, params);
    
    res.json(conversations);
  } catch (error) {
    console.error('Error fetching conversations:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) {
      connection.release();
    }
  }
});

// Fetch data from a specific conversation
router.get('/conversations/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const connection = await pool.getConnection();

    const [conversations] = await connection.execute(`
      SELECT c.*, u.username as assigned_user_name
      FROM conversations c
      LEFT JOIN users u ON c.assigned_user_id = u.id
      WHERE c.id = ?
    `, [id]);

    connection.release();
    
    if (conversations.length === 0) {
      return res.status(404).json({ error: 'Conversation not found' });
    }
    
    res.json(conversations[0]);
  } catch (error) {
    console.error('Error fetching conversation:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Fetch messages from a conversation
router.get('/conversations/:id/messages', async (req, res) => {
  try {
    const { id } = req.params;
    const connection = await pool.getConnection();

    const [messages] = await connection.execute(`
      SELECT m.*, u.username as sender_username
      FROM messages m
      LEFT JOIN users u ON m.sender_user_id = u.id
      WHERE m.conversation_id = ?
      ORDER BY m.timestamp ASC
    `, [id]);

    connection.release();
    res.json(messages);
  } catch (error) {
    console.error('Error fetching messages:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Attend conversation - MODIFIED
router.post('/conversations/:id/attend', async (req, res) => {
  let connection;
  try {
    const { id } = req.params;
    const { userId, username, store } = req.user;
    
    connection = await pool.getConnection();
    
    // Check if conversation is already being attended
    const [existing] = await connection.execute(
      'SELECT assigned_user_id, phone_number, assigned_store FROM conversations WHERE id = ?', 
      [id]
    );
    
    if (existing.length > 0 && existing[0].assigned_user_id) {
      return res.status(400).json({ error: 'Conversation is already being attended' });
    }
    
    const phoneNumber = existing[0].phone_number;
    const currentAssignedStore = existing[0].assigned_store;

    // Check if it's a department conversation (starts with SETOR: or DEPT:)
    const isFromDepartment = currentAssignedStore && (currentAssignedStore.startsWith('SETOR:') || currentAssignedStore.startsWith('DEPT:'));
    
    // If from department, keep assigned_store; if from store, update to user's store
    const newAssignedStore = isFromDepartment ? currentAssignedStore : store;

    // Update conversation - associate user but keep/update store according to context
    await connection.execute(
      'UPDATE conversations SET assigned_user_id = ?, assigned_store = ?, status = ? WHERE id = ?',
      [userId, newAssignedStore, 'attended', id]
    );
    
    const io = req.app.get('io');

    // Emit GLOBAL event to remove conversation from other stores (only if not from department)
    if (!isFromDepartment) {
      io.emit('conversation-attended', { 
        conversationId: id, 
        attendedBy: username, 
        store: store 
      });
    }
    
    res.json({ message: 'Conversation attended successfully' });
    
  } catch (error) {
    console.error('Error attending conversation:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Transfer conversation to store
router.post('/conversations/:id/transfer', async (req, res) => {
  let connection;
  try {
    const { id } = req.params;
    const { targetStore } = req.body;
    const { userId, username, store } = req.user;

    console.log('Transferring conversation:', { id, targetStore, fromUser: username, fromStore: store });

    if (!targetStore) {
      return res.status(400).json({ error: 'Target store is required' });
    }

    connection = await pool.getConnection();
    
    // Check if conversation exists
    const [conversations] = await connection.execute(
      'SELECT phone_number FROM conversations WHERE id = ?',
      [id]
    );

    if (conversations.length === 0) {
      connection.release();
      return res.status(404).json({ error: 'Conversation not found' });
    }

    const phoneNumber = conversations[0].phone_number;

    // Fetch store name by ID
    const [stores] = await connection.execute(
      'SELECT name FROM stores WHERE id = ?',
      [targetStore]
    );

    if (stores.length === 0) {
      connection.release();
      return res.status(400).json({ error: 'Store not found' });
    }

    const storeName = stores[0].name;

    // Update conversation to target store
    await connection.execute(
      'UPDATE conversations SET assigned_user_id = NULL, assigned_store = ?, status = ? WHERE id = ?',
      [storeName, 'waiting', id]
    );

    // Register system message about the transfer
    await connection.execute(
      `INSERT INTO messages (conversation_id, phone_number, message_text, message_type, is_from_bot, sender_user_id, timestamp)
       VALUES (?, ?, ?, ?, ?, ?, NOW())`,
      [
        id,
        phoneNumber,
        `🔄 Conversation transferred to store ${storeName} by ${username}${store ? ` (${store})` : ''}`,
        'text',
        true,
        userId
      ]
    );

    connection.release();

    // Notify users
    const io = req.app.get('io');
    if (io) {
      // Notify target store about new conversation
      io.to(storeName).emit('conversation-transferred', {
        conversationId: id,
        targetStore: storeName,
        from: store || 'support',
        transferredBy: username
      });

      // NEW: Notify origin store that conversation was removed
      if (store) {
        console.log(`Emitting conversation-removed to store: ${store}, conversation: ${id}`);
        io.to(store).emit('conversation-removed', {
          conversationId: parseInt(id),
          reason: 'transferred',
          targetStore: storeName
        });
      }
    }

    console.log('Conversation transferred successfully to store:', storeName);
    res.json({ message: 'Conversation transferred successfully' });
  } catch (error) {
    console.error('Error transferring conversation:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// Transfer conversation to department (support)
router.post('/conversations/:id/transfer-to-support', async (req, res) => {
  let connection;
  try {
    const { id } = req.params;
    const { targetDepartment } = req.body;
    const { userId, username, store } = req.user;

    if (!targetDepartment) {
      return res.status(400).json({ error: 'Target department is required' });
    }

    connection = await pool.getConnection();
    
    // Check if department exists
    const [departments] = await connection.execute(
      'SELECT id, name FROM departments WHERE name = ?',
      [targetDepartment]
    );

    if (departments.length === 0) {
      connection.release();
      return res.status(400).json({ error: 'Department not found' });
    }

    // Fetch conversation data
    const [conversations] = await connection.execute(
      'SELECT phone_number FROM conversations WHERE id = ?',
      [id]
    );

    if (conversations.length === 0) {
      connection.release();
      return res.status(404).json({ error: 'Conversation not found' });
    }

    const phoneNumber = conversations[0].phone_number;

    // Update conversation to department - use SETOR: prefix
    await connection.execute(
      'UPDATE conversations SET assigned_user_id = NULL, assigned_store = ?, status = ? WHERE id = ?',
      [`SETOR:${targetDepartment}`, 'waiting', id]
    );

    // Register system message about the transfer
    await connection.execute(
      `INSERT INTO messages (conversation_id, phone_number, message_text, message_type, is_from_bot, sender_user_id, timestamp)
       VALUES (?, ?, ?, ?, ?, ?, NOW())`,
      [
        id,
        phoneNumber,
        `📋 Conversation transferred to ${targetDepartment} by ${username} (${store})`,
        'text',
        true,
        userId
      ]
    );

    connection.release();

    // Notify users
    const io = req.app.get('io');
    if (io) {
      // Notify target department with SETOR: prefix
      const departmentRoom = `SETOR:${targetDepartment}`;
      console.log(`Emitting conversation-transferred to room: ${departmentRoom}`);
      io.to(departmentRoom).emit('conversation-transferred', {
        conversationId: id,
        targetDepartment: targetDepartment,
        from: store,
        transferredBy: username
      });

      // Notify origin store that conversation was removed
      if (store) {
        console.log(`Emitting conversation-removed to store: ${store}, conversation: ${id}`);
        io.to(store).emit('conversation-removed', {
          conversationId: parseInt(id),
          reason: 'transferred',
          targetDepartment: targetDepartment
        });
      }
    }

    res.json({ message: 'Conversation transferred to support successfully' });
  } catch (error) {
    console.error('Error transferring conversation to support:', error);
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    if (connection) connection.release();
  }
});

// List available departments
router.get('/departments', async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const [departments] = await connection.execute('SELECT id, name, description FROM departments ORDER BY name');
    connection.release();
    res.json(departments);
  } catch (error) {
    if (connection) connection.release();
    console.error('Error listing departments:', error);
    res.status(500).json({ error: 'Internal server error fetching departments' });
  }
});

// Replace the existing "Send message" route in your chat.js file with this version

// Send message - NO DUPLICATION
router.post('/conversations/:id/messages', async (req, res) => {
  try {
    const { id } = req.params;
    const { message, messageType = 'text', mediaUrl } = req.body;
    const { userId, username, store } = req.user;

    const connection = await pool.getConnection();

    // Fetch conversation number
    const [conversation] = await connection.execute(
      'SELECT phone_number, assigned_store FROM conversations WHERE id = ?',
      [id]
    );

    if (conversation.length === 0) {
      connection.release();
      return res.status(404).json({ error: 'Conversation not found' });
    }

    const phoneNumber = conversation[0].phone_number;
    connection.release();

    let finalMessage = message;
    let finalMediaUrl = mediaUrl;
    
    // If it's a text message, add user prefix
    if (messageType === 'text' && message) {
      let location = store;
      if (!store && conversation[0].assigned_store) {
        location = conversation[0].assigned_store.replace(/^(SETOR:|DEPT:)/, '');
      } else if (!store) {
        location = 'Support';
      }
      const userPrefix = `*${username} (${location}):*`;
      finalMessage = userPrefix + '\n' + message;
    }

    // DON'T SAVE TO DATABASE HERE - let whatsappService.saveOutgoingMessage do it
    // This avoids duplication

    // Send message via WhatsApp
    const whatsappService = req.app.get('whatsappService');
    if (whatsappService) {
      if (messageType === 'text') {
        await whatsappService.sendMessage(phoneNumber, finalMessage, { 
          senderUserId: userId,
          conversationId: parseInt(id)
        });
      } else if (finalMediaUrl) {
        await whatsappService.sendMessage(phoneNumber, { 
          mediaUrl: finalMediaUrl, 
          caption: finalMessage || '' 
        }, { 
          senderUserId: userId,
          conversationId: parseInt(id)
        });
      }
    }

    res.json({ message: 'Message sent successfully' });
  } catch (error) {
    console.error('Error sending message:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
});

// Route for file upload
const multer = require('multer');
const path = require('path');
const fs = require('fs');

// Configure multer for upload
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    const uploadDir = path.join(__dirname, '../uploads');
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }
    cb(null, uploadDir);
  },
  filename: function (req, file, cb) {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, uniqueSuffix + path.extname(file.originalname));
  }
});

const upload = multer({ 
  storage: storage,
  limits: { fileSize: 50 * 1024 * 1024 } // 50MB
});

router.post('/upload', upload.single('file'), async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ error: 'No file uploaded' });
    }
    
    const fileUrl = `/uploads/${req.file.filename}`;
    res.json({ url: fileUrl, filename: req.file.filename });
  } catch (error) {
    console.error('Error uploading file:', error);
    res.status(500).json({ error: 'Error uploading file' });
  }
});

module.exports = router;