// ===== GLOBAL VARIABLES =====
// API_BASE is already declared in auth.js
let socket;
let currentView = 'conversations';
let currentConversationId = null;
let currentConversation = null;
let mediaRecorder = null;
let audioChunks = [];
let isRecording = false;
let conversations = [];
let recordingPausedTime = 0;
let recordingPausedAt = null;
let recordingCancelled = false;
let isSendingMessage = false;

// ===== INITIALIZATION =====
document.addEventListener('DOMContentLoaded', function() {
    console.log('Initializing application...');
    
    if (!requireAuth()) return;
    
    const user = getCurrentUser();
    if (!user) {
        console.error('User not found');
        return;
    }
    
    if (user.role === 'admin') {
        window.location.href = '/admin';
        return;
    }
    
    initializeApp();
});

// ===== AUTHENTICATION =====
function requireAuth() {
    const token = localStorage.getItem('authToken');
    const userStr = localStorage.getItem('currentUser');
    
    if (!token || !userStr) {
        window.location.href = '/';
        return false;
    }
    
    return true;
}

function getCurrentUser() {
    const userStr = localStorage.getItem('currentUser');
    return userStr ? JSON.parse(userStr) : null;
}

// ===== LOAD SYSTEM SETTINGS =====
async function loadSystemSettings() {
    try {
        const response = await fetch('/api/public/profile-settings');
        if (response.ok) {
            const settings = await response.json();
            
            // Update page title
            if (settings.systemName) {
                document.getElementById('pageTitle').textContent = settings.systemName + ' - Service';
            }
            
            // Update favicon
            if (settings.favicon) {
                document.getElementById('pageFavicon').href = settings.favicon;
            }
        }
    } catch (error) {
        console.error('Error loading system settings:', error);
    }
}

// ===== INITIALIZE APP =====
async function initializeApp() {
    try {
        // Load system settings (favicon and name)
        await loadSystemSettings();
        
        // Load saved theme
        const savedTheme = localStorage.getItem('theme') || 'light';
        document.documentElement.setAttribute('data-theme', savedTheme);
        
        // Load active system language
        if (typeof i18n !== 'undefined' && i18n.loadLanguage) {
            await i18n.loadLanguage();
        }
        
        // Initialize Socket.IO
        initializeSocket();
        
        // Load conversations
        await loadConversations();
        
        // Setup event listeners
        setupEventListeners();
        
        // Initialize buttons
        initializeButtons();
        
        console.log('Application initialized successfully!');
    } catch (error) {
        console.error('Error initializing:', error);
        showNotification(i18n.t('user.error_init_app'), 'error');
    }
}

// ===== SOCKET.IO =====
function initializeSocket() {
    socket = io({
        timeout: 20000,
        reconnection: true,
        reconnectionDelay: 2000,
        reconnectionAttempts: 5,
        transports: ['websocket', 'polling']
    });
    
    socket.on('connect', async () => {
        console.log('Socket conectado');
        const user = getCurrentUser();
        if (user && user.store) {
            socket.emit('join-room', user.store);
        } else {
            // User sem loja (setor/suporte): entrar em todas as salas de departamentos
            try {
                const response = await authenticatedFetch(`${API_BASE}/api/chat/departments`);
                const departments = await response.json();
                departments.forEach(dept => {
                    socket.emit('join-department', dept.name);
                    console.log(`Entrando na sala do departamento: ${dept.name}`);
                });
            } catch (error) {
                console.error('Error carregar departamentos para socket:', error);
            }
        }
    });
    
    socket.on('new-message', (data) => {
        console.log('Nova mensagem recebida:', data);
        handleNewMessage(data);
    });
    
    socket.on('conversation-updated', (data) => {
        console.log('Conversa atualizada:', data);
        loadConversations();
    });
    
    socket.on('conversation-attended', (data) => {
        console.log('Conversa atendida:', data);
        const user = getCurrentUser();
        // Se não for da minha loja, remover da lista
        if (user.store !== data.store) {
            handleConversationRemoved(data);
        }
    });
    
    socket.on('conversation-removed', (data) => {
        console.log('Conversa removida:', data);
        handleConversationRemoved(data);
    });
    
    socket.on('conversation-transferred', (data) => {
        console.log('Conversa transferida:', data);
        loadConversations();
    });
    
    socket.on('disconnect', () => {
        console.log('Socket desconectado');
    });
}

// ===== HANDLER PARA CONVERSA REMOVIDA =====
function handleConversationRemoved(data) {
    // Se estiver visualizando a conversa que foi removida, voltar para lista
    if (currentConversationId === data.conversationId) {
        closeConversation();
        showNotification(`Conversa transferida para ${data.targetStore || data.targetDepartment}`, 'info');
    }
    
    // Recarregar lista de conversas
    loadConversations();
}

// ===== EVENT LISTENERS =====
function setupEventListeners() {
    // Input de mensagem
    const messageInput = document.getElementById('messageInput');
    if (messageInput) {
        messageInput.addEventListener('input', handleInputChange);
        messageInput.addEventListener('keypress', (e) => {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                sendMessage();
            }
        });
    }
    
    // File inputs
    document.getElementById('fileInput')?.addEventListener('change', handleFileSelect);
    document.getElementById('imageInput')?.addEventListener('change', handleFileSelect);
    document.getElementById('videoInput')?.addEventListener('change', handleFileSelect);
    document.getElementById('documentInput')?.addEventListener('change', handleFileSelect);
    
    // Fechar menu ao clicar fora
    document.addEventListener('click', (e) => {
        const menuDropdown = document.getElementById('menuDropdown');
        if (menuDropdown && !e.target.closest('.icon-btn') && !e.target.closest('.menu-dropdown')) {
            menuDropdown.style.display = 'none';
        }
    });
}

function handleInputChange() {
    const messageInput = document.getElementById('messageInput');
    const sendBtn = document.getElementById('sendBtn');
    const micBtn = document.getElementById('micBtn');
    
    if (messageInput.value.trim()) {
        sendBtn.style.display = 'flex';
        micBtn.style.display = 'none';
    } else {
        sendBtn.style.display = 'none';
        micBtn.style.display = 'flex';
    }
}

// Inicializar botão microfone como padrão
function initializeButtons() {
    const sendBtn = document.getElementById('sendBtn');
    const micBtn = document.getElementById('micBtn');
    
    sendBtn.style.display = 'none';
    micBtn.style.display = 'flex';
}

// ===== TEMA =====
window.toggleTheme = function() {
    const currentTheme = document.documentElement.getAttribute('data-theme');
    const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
    
    document.documentElement.setAttribute('data-theme', newTheme);
    localStorage.setItem('theme', newTheme);
    
    showNotification(`${i18n.t(newTheme === 'dark' ? 'user.theme_dark_activated' : 'user.theme_light_activated')}`, 'success');
}

// ===== BUSCA =====
window.toggleSearch = function() {
    const searchBar = document.getElementById('searchBar');
    const searchInput = document.getElementById('searchInput');
    
    if (searchBar.style.display === 'none' || !searchBar.style.display) {
        searchBar.style.display = 'block';
        searchInput.focus();
    } else {
        searchBar.style.display = 'none';
        searchInput.value = '';
        filterConversations('');
    }
}

window.clearSearch = function() {
    const searchInput = document.getElementById('searchInput');
    searchInput.value = '';
    filterConversations('');
    searchInput.focus();
}

document.getElementById('searchInput')?.addEventListener('input', (e) => {
    filterConversations(e.target.value);
});

function filterConversations(query) {
    const items = document.querySelectorAll('.conversation-item');
    const lowerQuery = query.toLowerCase();
    
    items.forEach(item => {
        const name = item.querySelector('.conversation-name')?.textContent.toLowerCase() || '';
        const phone = item.querySelector('.conversation-phone')?.textContent.toLowerCase() || '';
        const preview = item.querySelector('.conversation-preview')?.textContent.toLowerCase() || '';
        
        if (name.includes(lowerQuery) || phone.includes(lowerQuery) || preview.includes(lowerQuery)) {
            item.style.display = 'flex';
        } else {
            item.style.display = 'none';
        }
    });
}

// ===== LOGOUT =====
window.openLogoutModal = function() {
    document.getElementById('logoutModal').style.display = 'flex';
}

window.closeLogoutModal = function() {
    document.getElementById('logoutModal').style.display = 'none';
}

window.confirmLogout = function() {
    closeLogoutModal();
    logout();
}

window.logout = function() {
    localStorage.removeItem('authToken');
    localStorage.removeItem('currentUser');
    window.location.href = '/';
}

window.showMassMessageView = function() {
    // Redirecionar para página de envio em massa ou abrir modal
    showNotification(i18n.t('user.mass_send_development'), 'info');
}

// ===== CONVERSAS =====
async function loadConversations() {
    try {
        const response = await authenticatedFetch(`${API_BASE}/api/chat/conversations`);
        const data = await response.json();
        
        // Verificar se a resposta é um array ou se contém erro
        if (Array.isArray(data)) {
            conversations = data;
        } else if (data.error) {
            console.error('Erro do servidor:', data.error);
            showNotification(data.error, 'error');
            conversations = [];
        } else {
            console.error('Resposta inesperada do servidor:', data);
            conversations = [];
        }
        
        renderConversations(conversations);
    } catch (error) {
        console.error('Error carregar conversas:', error);
        showNotification(i18n.t('user.error_load_conversations'), 'error');
        conversations = [];
        renderConversations(conversations);
    }
}

function renderConversations(convs) {
    const container = document.getElementById('conversationsList');
    
    if (!convs || convs.length === 0) {
        container.innerHTML = `
            <div class="no-conversations">
                <svg viewBox="0 0 24 24"><path fill="currentColor" d="M20 2H4c-1.1 0-2 .9-2 2v18l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2z"/></svg>
                <p data-i18n='user.no_conversations'></p>
            </div>
        `;
        return;
    }
    
    container.innerHTML = convs.map(conv => {
        const initials = (conv.contact_name || conv.phone_number).substring(0, 2).toUpperCase();
        
        // Determinar o preview da mensagem baseado no tipo
        let lastMessage = conv.last_message || '';
        
        // Se não há texto mas há tipo de mensagem, mostrar descrição da mídia
        if (!lastMessage && conv.last_message_type) {
            switch(conv.last_message_type) {
                case 'image':
                    lastMessage = i18n.t('user.media_photo');
                    break;
                case 'video':
                    lastMessage = i18n.t('user.media_video');
                    break;
                case 'audio':
                    lastMessage = i18n.t('user.media_audio');
                    break;
                case 'document':
                    lastMessage = i18n.t('user.media_document');
                    break;
                case 'location':
                    lastMessage = i18n.t('user.media_location');
                    break;
                default:
                    lastMessage = i18n.t('user.media_message');
            }
        }
        
        // Se ainda não há mensagem, mostrar i18n.t('user.no_messages')
        if (!lastMessage) {
            lastMessage = i18n.t('user.no_messages');
        }
        
        const time = formatTime(conv.last_message_time);
        
        return `
            <div class="conversation-item" onclick="openConversation(${conv.id})">
                <div class="conversation-avatar">${initials}</div>
                <div class="conversation-info">
                    <div class="conversation-header">
                        <span class="conversation-name">${conv.contact_name || conv.phone_number}</span>
                        <span class="conversation-time">${time}</span>
                    </div>
                    <div class="conversation-preview">${lastMessage}</div>
                    <span class="conversation-phone" style="display:none;">${conv.phone_number}</span>
                </div>
            </div>
        `;
    }).join('');
}

window.openConversation = async function(conversationId) {
    try {
        currentConversationId = conversationId;
        
        // Search dados da conversa
        const response = await authenticatedFetch(`${API_BASE}/api/chat/conversations/${conversationId}`);
        currentConversation = await response.json();
        
        // Auto-atendimento: se a conversa não está atendida, atender automaticamente
        // Mas apenas se for usuário de LOJA (não mudar assigned_store de setores)
        const currentUser = getCurrentUser();
        if (!currentConversation.assigned_user_id && currentUser.store) {
            try {
                await authenticatedFetch(`${API_BASE}/api/chat/conversations/${conversationId}/attend`, {
                    method: 'POST'
                });
            } catch (attendError) {
                console.error('Error atender conversa automaticamente:', attendError);
            }
        }
        
        // Atualizar header
        const headerTitle = document.getElementById('headerTitle');
        const headerSubtitle = document.getElementById('headerSubtitle');
        headerTitle.textContent = currentConversation.contact_name || currentConversation.phone_number;
        headerSubtitle.textContent = currentConversation.phone_number;
        
        // Mostrar botão voltar
        document.getElementById('backBtn').style.display = 'flex';
        
        // Mostrar/esconder botões corretos
        document.getElementById('chatMenuBtn').style.display = 'flex';
        document.getElementById('mainMenuBtn').style.display = 'none';
        
        // Esconder botão de contatos quando estiver em uma conversa
        const contactsBtn = document.querySelector('.floating-contacts-btn');
        if (contactsBtn) {
            contactsBtn.style.display = 'none';
        }
        
        // Trocar view
        document.getElementById('conversationsView').style.display = 'none';
        document.getElementById('chatView').style.display = 'flex';
        
        // Limpar mensagens antigas e carregar novas
        currentMessages.clear();
        await loadMessages(conversationId);
        
        // Focar no input
        document.getElementById('messageInput').focus();
        
    } catch (error) {
        console.error('Error abrir conversa:', error);
        showNotification(i18n.t('user.error_open_conversation'), 'error');
    }
}

function closeConversation() {
    currentConversationId = null;
    currentConversation = null;
    currentMessages.clear(); // Limpar mensagens ao fechar conversa
    
    // Atualizar header
    document.getElementById('headerTitle').textContent = 'Conversas';
    document.getElementById('headerSubtitle').textContent = '';
    document.getElementById('backBtn').style.display = 'none';
    
    // Mostrar/esconder botões corretos
    document.getElementById('chatMenuBtn').style.display = 'none';
    document.getElementById('mainMenuBtn').style.display = 'flex';
    
    // Mostrar botão de contatos quando voltar para lista de conversas
    const contactsBtn = document.querySelector('.floating-contacts-btn');
    if (contactsBtn) {
        contactsBtn.style.display = 'block';
    }
    
    // Trocar view
    document.getElementById('conversationsView').style.display = 'block';
    document.getElementById('chatView').style.display = 'none';
    
    // Recarregar conversas
    loadConversations();
}

document.getElementById('backBtn')?.addEventListener('click', closeConversation);

// ===== MENSAGENS =====
async function loadMessages(conversationId) {
    console.log('[LOAD] Loading mensagens para conversa:', conversationId);
    try {
        const response = await authenticatedFetch(`${API_BASE}/api/chat/conversations/${conversationId}/messages`);
        const messages = await response.json();
        
        console.log('[LOAD] Messages recebidas da API:', messages.length);
        renderMessages(messages);
    } catch (error) {
        console.error('Error carregar mensagens:', error);
        showNotification(i18n.t('user.error_load_messages'), 'error');
    }
}

let currentMessages = new Map(); // Armazena mensagens por ID

function renderMessages(messages, append = false) {
    const container = document.getElementById('messagesContainer');
    
    console.log('[RENDER] Total de mensagens recebidas:', messages.length, 'Append:', append);
    
    if (!messages || messages.length === 0) {
        if (!append) {
            container.innerHTML = `
                <div class="no-messages">
                    <p data-i18n='user.no_messages_conversation'></p>
                </div>
            `;
            currentMessages.clear();
        }
        return;
    }
    
    // Se não for append, limpar mensagens atuais
    if (!append) {
        currentMessages.clear();
    }
    
    // Add/atualizar mensagens no Map
    messages.forEach(msg => {
        if (!currentMessages.has(msg.id)) {
            currentMessages.set(msg.id, msg);
        } else {
            console.log('[RENDER] Message já existe, ignorando:', msg.id);
        }
    });
    
    // Converter Map para array e ordenar por timestamp
    const uniqueMessages = Array.from(currentMessages.values())
        .sort((a, b) => new Date(a.timestamp) - new Date(b.timestamp));
    
    console.log('[RENDER] Total de mensagens únicas:', uniqueMessages.length);
    
    container.innerHTML = uniqueMessages.map(msg => {
        const isSent = msg.is_from_bot || msg.sender_user_id;
        const time = formatTime(msg.timestamp);
        
        let content = msg.message_text || '';
        
        // Tratar mídia
        if (msg.media_url) {
            if (msg.message_type === 'image') {
                content = `<img src="${msg.media_url}" alt="Imagem" style="max-width: 300px; border-radius: 8px;">`;
            } else if (msg.message_type === 'video') {
                content = `<video controls style="max-width: 300px; border-radius: 8px;"><source src="${msg.media_url}"></video>`;
            } else if (msg.message_type === 'audio') {
                // Gerar waveform visual (barras aleatórias para simular espectro)
                const waveformBars = Array.from({length: 35}, (_, i) => {
                    const height = Math.random() * 20 + 4; // altura entre 4px e 24px
                    return `<div class="waveform-bar" style="height: ${height}px;"></div>`;
                }).join('');
                
                content = `
                    <div class="audio-message">
                        <div class="audio-content">
                            <div class="audio-controls">
                                <button class="audio-play-btn" onclick="toggleAudio(this, '${msg.media_url}')">
                                    <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M8 5v14l11-7z"/>
                                    </svg>
                                </button>
                                <div class="audio-waveform-container">
                                    <div class="audio-waveform" onclick="seekAudio(this, event)">
                                        ${waveformBars}
                                    </div>
                                    <div class="audio-time-info">
                                        <span class="audio-current-time">0:00</span>
                                        <span class="audio-duration">
                                            <span id="duration-${msg.id}">--:--</span>
                                            <svg width="12" height="12" viewBox="0 0 24 24" fill="currentColor">
                                                <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z"/>
                                            </svg>
                                        </span>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <audio style="display: none;" src="${msg.media_url}" onloadedmetadata="updateAudioDuration(this, 'duration-${msg.id}')"></audio>
                    </div>
                `;
            } else if (msg.message_type === 'document') {
                content = `<a href="${msg.media_url}" target="_blank">📄 ${msg.message_text || 'Documento'}</a>`;
            }
        }
        
        // Tratar localização
        if (msg.message_type === 'location' && msg.location_data) {
            const locationData = typeof msg.location_data === 'string' ? JSON.parse(msg.location_data) : msg.location_data;
            const mapsUrl = locationData.googleMapsUrl || `https://www.google.com/maps?q=${locationData.lat},${locationData.lng}`;
            content = `📍 <a href="${mapsUrl}" target="_blank" style="color: #4fc3f7; text-decoration: underline;">${locationData.address || 'Ver localização no mapa'}</a>`;
        }
        
        return `
            <div class="message ${isSent ? 'sent' : 'received'}">
                <div class="message-bubble">
                    ${msg.sender_username ? `<div class="message-sender">${msg.sender_username}</div>` : ''}
                    <div class="message-text">${content}</div>
                    <div class="message-time">${time}</div>
                </div>
            </div>
        `;
    }).join('');
    
    // Scroll para o final
    setTimeout(() => {
        container.scrollTop = container.scrollHeight;
    }, 100);
}

// ===== CONTROLE DE ÁUDIO =====
window.toggleAudio = function(button, audioUrl) {
    const audioMessage = button.closest('.audio-message');
    const audio = audioMessage.querySelector('audio');
    const playIcon = button.querySelector('svg path');
    const currentTimeDisplay = audioMessage.querySelector('.audio-current-time');
    
    if (audio.paused) {
        // Pausar todos os outros áudios
        document.querySelectorAll('.audio-message audio').forEach(otherAudio => {
            if (otherAudio !== audio) {
                otherAudio.pause();
                const otherButton = otherAudio.closest('.audio-message').querySelector('.audio-play-btn svg path');
                otherButton.setAttribute('d', 'M8 5v14l11-7z'); // Play icon
            }
        });
        
        audio.play();
        playIcon.setAttribute('d', 'M6 19h4V5H6v14zm8-14v14h4V5h-4z'); // Pause icon
        
        // Atualizar tempo atual
        audio.ontimeupdate = () => {
            const minutes = Math.floor(audio.currentTime / 60);
            const seconds = Math.floor(audio.currentTime % 60);
            currentTimeDisplay.textContent = `${minutes}:${seconds.toString().padStart(2, '0')}`;
            
            // Atualizar progresso visual
            const progress = (audio.currentTime / audio.duration) * 100;
            const waveformBars = audioMessage.querySelectorAll('.waveform-bar');
            waveformBars.forEach((bar, index) => {
                const barProgress = (index / waveformBars.length) * 100;
                if (barProgress <= progress) {
                    bar.style.backgroundColor = '#00d4aa';
                } else {
                    bar.style.backgroundColor = '#8696A0';
                }
            });
        };
        
        audio.onended = () => {
            playIcon.setAttribute('d', 'M8 5v14l11-7z'); // Play icon
            currentTimeDisplay.textContent = '0:00';
            // Reset visual progress
            audioMessage.querySelectorAll('.waveform-bar').forEach(bar => {
                bar.style.backgroundColor = '#8696A0';
            });
        };
    } else {
        audio.pause();
        playIcon.setAttribute('d', 'M8 5v14l11-7z'); // Play icon
    }
};

window.seekAudio = function(waveform, event) {
    const audioMessage = waveform.closest('.audio-message');
    const audio = audioMessage.querySelector('audio');
    const rect = waveform.getBoundingClientRect();
    const clickX = event.clientX - rect.left;
    const percentage = clickX / rect.width;
    
    if (audio.duration) {
        audio.currentTime = audio.duration * percentage;
    }
};

window.openLocationInMaps = function(lat, lng) {
    const url = `https://www.google.com/maps?q=${lat},${lng}`;
    window.open(url, '_blank');
};

window.sendMessage = async function() {
    const input = document.getElementById('messageInput');
    const message = input.value.trim();
    
    if (!message || !currentConversationId) return;
    
    // Prevenir envios duplicados
    if (isSendingMessage) {
        console.log('[SEND] Já existe um envio em andamento, ignorando...');
        return;
    }
    
    console.log('[SEND] Enviando mensagem:', message);
    
    // Marcar que está enviando
    isSendingMessage = true;
    
    // Limpar input imediatamente para evitar envios duplicados
    input.value = '';
    handleInputChange();
    
    try {
        const response = await authenticatedFetch(`${API_BASE}/api/chat/conversations/${currentConversationId}/messages`, {
            method: 'POST',
            body: JSON.stringify({
                message: message,
                messageType: 'text'
            })
        });
        
        if (!response.ok) {
            // Se deu erro, restaurar a mensagem no input
            input.value = message;
            handleInputChange();
            throw new Error(i18n.t('user.error_send_message'));
        }
        
        console.log('[SEND] Message enviada successfully');
        
    } catch (error) {
        console.error('Error enviar mensagem:', error);
        showNotification(i18n.t('user.error_send_message'), 'error');
    } finally {
        // Liberar flag de envio
        isSendingMessage = false;
    }
}

let lastMessageId = null;
let loadMessagesTimeout = null;

function handleNewMessage(data) {
    console.log('[SOCKET] Nova mensagem recebida via socket:', data);
    
    // Evitar processar a mesma mensagem múltiplas vezes
    if (data.messageId && data.messageId === lastMessageId) {
        console.log('[SOCKET] Message já processada, ignorando:', data.messageId);
        return;
    }
    console.log('[SOCKET] Processando nova mensagem:', data.messageId);
    lastMessageId = data.messageId;
    
    // Se estiver na conversa, recarregar mensagens com debounce
    if (currentConversationId === data.conversationId) {
        if (loadMessagesTimeout) {
            clearTimeout(loadMessagesTimeout);
        }
        loadMessagesTimeout = setTimeout(() => {
            loadMessages(currentConversationId);
            loadMessagesTimeout = null;
        }, 300);
    }
    
    // Atualizar lista de conversas
    loadConversations();
    
    // Mostrar notificação
    if (Notification.permission === 'granted') {
        const messageText = data.message || (data.messageType === 'image' ? i18n.t('user.media_photo') : 
                                           data.messageType === 'video' ? i18n.t('user.media_video') :
                                           data.messageType === 'audio' ? i18n.t('user.media_audio') :
                                           data.messageType === 'document' ? i18n.t('user.media_document') : 'Nova mensagem');
        new Notification('Nova mensagem', {
            body: messageText,
            icon: '/icon-192x192.png'
        });
    }
}

// ===== MENU DE ANEXOS =====
window.toggleAttachMenu = function() {
    const menu = document.getElementById('attachMenu');
    menu.style.display = menu.style.display === 'none' ? 'block' : 'none';
}

window.selectAttachment = function(type) {
    toggleAttachMenu();
    
    const fileInput = document.getElementById('fileInput');
    
    switch(type) {
        case 'gallery':
            fileInput.accept = 'image/*,video/*';
            fileInput.multiple = false;
            fileInput.click();
            break;
        case 'camera':
            fileInput.accept = 'image/*';
            fileInput.capture = 'environment';
            fileInput.multiple = false;
            fileInput.click();
            break;
        case 'location':
            openLocationModal();
            break;
        case 'contact':
            showNotification(i18n.t('user.contact_development'), 'info');
            break;
        case 'document':
            fileInput.accept = '.pdf,.doc,.docx,.xls,.xlsx,.txt,.zip,.rar';
            fileInput.multiple = false;
            fileInput.click();
            break;
        case 'audio':
            fileInput.accept = 'audio/*';
            fileInput.multiple = false;
            fileInput.click();
            break;
        case 'payment':
            openPaymentModal();
            break;
    }
}

// ===== MÍDIA =====
async function handleFileSelect(e) {
    const file = e.target.files[0];
    if (!file || !currentConversationId) return;
    
    try {
        const formData = new FormData();
        formData.append('file', file);
        formData.append('conversationId', currentConversationId);
        
        const response = await fetch(`${API_BASE}/api/chat/upload`, {
            method: 'POST',
            headers: {
                'Authorization': `Bearer ${localStorage.getItem('authToken')}`
            },
            body: formData
        });
        
        if (response.ok) {
            const data = await response.json();
            await sendMediaMessage(data.url, file.type);
        } else {
            throw new Error('Error fazer upload');
        }
    } catch (error) {
        console.error('Error enviar arquivo:', error);
        showNotification(i18n.t('user.error_send_file'), 'error');
    }
    
    e.target.value = '';
}

async function sendMediaMessage(mediaUrl, mimeType) {
    let messageType = 'document';
    
    if (mimeType.startsWith('image/')) messageType = 'image';
    else if (mimeType.startsWith('video/')) messageType = 'video';
    else if (mimeType.startsWith('audio/')) messageType = 'audio';
    
    try {
        const response = await authenticatedFetch(`${API_BASE}/api/chat/conversations/${currentConversationId}/messages`, {
            method: 'POST',
            body: JSON.stringify({
                message: '',
                messageType: messageType,
                mediaUrl: mediaUrl
            })
        });
        
        if (response.ok) {
            // REMOVIDO: await loadMessages(currentConversationId);
            // O socket já vai receber a nova mensagem e atualizar automaticamente
        }
    } catch (error) {
        console.error('Error enviar mídia:', error);
        showNotification(i18n.t('user.error_send_media'), 'error');
    }
}// ===== LOCALIZAÇÃO =====
let map;
let marker;
let selectedLocation = null;

window.openLocationModal = function() {
    document.getElementById('locationModal').style.display = 'flex';
    
    // Inicializar mapa se ainda não foi inicializado
    setTimeout(() => {
        if (!map) {
            initializeMap();
        }
    }, 100);
};

window.closeLocationModal = function() {
    document.getElementById('locationModal').style.display = 'none';
};

function initializeMap() {
    const defaultLocation = [-23.5505, -46.6333]; // São Paulo [lat, lng]
    
    // Criar mapa usando Leaflet
    map = L.map('map').setView(defaultLocation, 15);
    
    // Add tiles do OpenStreetMap
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '© OpenStreetMap contributors'
    }).addTo(map);
    
    // Add marcador arrastável
    marker = L.marker(defaultLocation, { draggable: true }).addTo(map);
    
    // Atualizar informações quando o marcador for movido
    marker.on('dragend', function(e) {
        const position = e.target.getLatLng();
        updateLocationInfo(position.lat, position.lng);
    });
    
    // Atualizar informações iniciais
    updateLocationInfo(defaultLocation[0], defaultLocation[1]);
    
    // Tentar obter localização atual
    if (navigator.geolocation) {
        navigator.geolocation.getCurrentPosition(
            (position) => {
                const userLocation = [position.coords.latitude, position.coords.longitude];
                map.setView(userLocation, 15);
                marker.setLatLng(userLocation);
                updateLocationInfo(userLocation[0], userLocation[1]);
            },
            (error) => {
                console.log('Error obter localização:', error);
            }
        );
    }
}

window.searchLocation = function() {
    const searchText = document.getElementById('locationSearch').value;
    if (!searchText) return;
    
    // Usar Nominatim para geocoding (OpenStreetMap)
    fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(searchText)}`)
        .then(response => response.json())
        .then(data => {
            if (data && data.length > 0) {
                const result = data[0];
                const lat = parseFloat(result.lat);
                const lng = parseFloat(result.lon);
                
                map.setView([lat, lng], 15);
                marker.setLatLng([lat, lng]);
                updateLocationInfo(lat, lng);
            } else {
                showNotification(i18n.t('user.error_location_not_found'), 'error');
            }
        })
        .catch(error => {
            console.error('Error in busca:', error);
            showNotification(i18n.t('user.error_get_location'), 'error');
        });
};

function updateLocationInfo(lat, lng) {
    selectedLocation = { lat, lng };
    
    document.getElementById('selectedCoords').value = `${lat.toFixed(6)}, ${lng.toFixed(6)}`;
    
    // Search endereço usando geocoding reverso do Nominatim
    fetch(`https://nominatim.openstreetmap.org/reverse?format=json&lat=${lat}&lon=${lng}`)
        .then(response => response.json())
        .then(data => {
            if (data && data.display_name) {
                document.getElementById('selectedAddress').value = data.display_name;
            }
        })
        .catch(error => {
            console.error('Error in geocoding reverso:', error);
            document.getElementById('selectedAddress').value = `${lat.toFixed(6)}, ${lng.toFixed(6)}`;
        });
}

window.sendLocation = function() {
    if (!selectedLocation) {
        showNotification(i18n.t('user.select_location'), 'error');
        return;
    }
    
    const address = document.getElementById('selectedAddress').value;
    const coords = document.getElementById('selectedCoords').value;
    
    // Criar link do Google Maps
    const locationData = {
        lat: selectedLocation.lat,
        lng: selectedLocation.lng,
        address: address,
        googleMapsUrl: `https://www.google.com/maps?q=${selectedLocation.lat},${selectedLocation.lng}`
    };
    
    sendLocationMessage(locationData);
    closeLocationModal();
};

async function sendLocationMessage(locationData) {
    try {
        const response = await authenticatedFetch(`${API_BASE}/api/chat/conversations/${currentConversationId}/messages`, {
            method: 'POST',
            body: JSON.stringify({
                message: `📍 ${locationData.address}`,
                messageType: 'location',
                locationData: locationData
            })
        });
        
        if (response.ok) {
            // REMOVIDO: await loadMessages(currentConversationId);
            // O socket já vai receber a nova mensagem e atualizar automaticamente
            showNotification(i18n.t('user.location_sent'), 'success');
        } else {
            showNotification(i18n.t('user.error_send_location'), 'error');
        }
    } catch (error) {
        console.error('Error enviar localização:', error);
        showNotification(i18n.t('user.error_send_location'), 'error');
    }
}

// ===== GRAVAÇÃO DE ÁUDIO =====
function showRecordingInterface() {
    const recordingUI = document.getElementById('recordingInterface');
    if (recordingUI) {
        recordingUI.style.display = 'flex';
        startRecording();
    }
}

function hideRecordingInterface() {
    const recordingUI = document.getElementById('recordingInterface');
    if (recordingUI) {
        recordingUI.style.display = 'none';
    }
}

window.startRecording = async function() {
    try {
        audioStream = await navigator.mediaDevices.getUserMedia({ audio: true });
        mediaRecorder = new MediaRecorder(audioStream);
        audioChunks = [];
        
        // Configure análise de áudio para visualização
        audioContext = new (window.AudioContext || window.webkitAudioContext)();
        analyser = audioContext.createAnalyser();
        const source = audioContext.createMediaStreamSource(audioStream);
        source.connect(analyser);
        analyser.fftSize = 256;
        
        mediaRecorder.ondataavailable = (e) => {
            audioChunks.push(e.data);
        };
        
        mediaRecorder.onstop = async () => {
            // Só enviar se NÃO foi cancelado
            if (!recordingCancelled && audioChunks.length > 0) {
                const audioBlob = new Blob(audioChunks, { type: 'audio/webm' });
                await uploadAudio(audioBlob);
            }
            cleanupRecording();
        };
        
        mediaRecorder.start();
        isRecording = true;
        recordingStartTime = Date.now();
        recordingPausedTime = 0;
        recordingPausedAt = null;
        recordingCancelled = false;
        
        // Iniciar timer
        updateRecordingTimer();
        
        // Iniciar visualização
        drawAudioSpectrum();
        
        // Atualizar UI
        const recordingUI = document.getElementById('recordingInterface');
        if (recordingUI) {
            recordingUI.style.display = 'flex';
            document.getElementById('pauseRecordingBtn').style.display = 'inline-flex';
            document.getElementById('resumeRecordingBtn').style.display = 'none';
        }
        
    } catch (error) {
        console.error('Error iniciar gravação:', error);
        showNotification(i18n.t('user.error_microphone_access'), 'error');
        hideRecordingInterface();
    }
}

function updateRecordingTimer() {
    recordingTimer = setInterval(() => {
        const now = Date.now();
        const totalElapsed = now - recordingStartTime;
        const elapsed = Math.floor((totalElapsed - recordingPausedTime) / 1000);
        const minutes = Math.floor(elapsed / 60);
        const seconds = elapsed % 60;
        const timerEl = document.getElementById('recordingTimer');
        if (timerEl) {
            timerEl.textContent = `${minutes}:${seconds.toString().padStart(2, '0')}`;
        }
    }, 1000);
}

function drawAudioSpectrum() {
    if (!isRecording || !analyser) return;
    
    const canvas = document.getElementById('audioSpectrum');
    if (!canvas) return;
    
    const ctx = canvas.getContext('2d');
    const bufferLength = analyser.frequencyBinCount;
    const dataArray = new Uint8Array(bufferLength);
    
    const draw = () => {
        if (!isRecording) return;
        
        animationFrame = requestAnimationFrame(draw);
        
        analyser.getByteFrequencyData(dataArray);
        
        ctx.fillStyle = 'rgba(0, 0, 0, 0)';
        ctx.clearRect(0, 0, canvas.width, canvas.height);
        
        const barWidth = 3;
        const gap = 2;
        const barCount = 50;
        let x = 0;
        
        for (let i = 0; i < barCount; i++) {
            const barHeight = (dataArray[i] / 255) * canvas.height * 0.8;
            const y = (canvas.height - barHeight) / 2;
            
            ctx.fillStyle = '#8696A0';
            ctx.fillRect(x, y, barWidth, barHeight);
            
            x += barWidth + gap;
        }
    };
    
    draw();
}

window.pauseRecording = function() {
    if (mediaRecorder && mediaRecorder.state === 'recording') {
        mediaRecorder.pause();
        recordingPausedAt = Date.now();
        clearInterval(recordingTimer);
        document.getElementById('pauseRecordingBtn').style.display = 'none';
        document.getElementById('resumeRecordingBtn').style.display = 'inline-flex';
    }
}

window.resumeRecording = function() {
    if (mediaRecorder && mediaRecorder.state === 'paused') {
        mediaRecorder.resume();
        if (recordingPausedAt) {
            recordingPausedTime += Date.now() - recordingPausedAt;
            recordingPausedAt = null;
        }
        updateRecordingTimer();
        document.getElementById('pauseRecordingBtn').style.display = 'inline-flex';
        document.getElementById('resumeRecordingBtn').style.display = 'none';
    }
}

window.cancelRecording = function() {
    if (mediaRecorder && isRecording) {
        // Marcar como cancelado ANTES de parar
        recordingCancelled = true;
        audioChunks = [];
        mediaRecorder.stop();
        isRecording = false;
        hideRecordingInterface();
        showNotification(i18n.t('user.recording_cancelled'), 'info');
    }
}

window.sendRecording = function() {
    if (mediaRecorder && isRecording) {
        mediaRecorder.stop();
        isRecording = false;
        hideRecordingInterface();
    }
}

function cleanupRecording() {
    if (audioStream) {
        audioStream.getTracks().forEach(track => track.stop());
        audioStream = null;
    }
    if (audioContext) {
        audioContext.close();
        audioContext = null;
    }
    if (recordingTimer) {
        clearInterval(recordingTimer);
        recordingTimer = null;
    }
    if (animationFrame) {
        cancelAnimationFrame(animationFrame);
        animationFrame = null;
    }
    analyser = null;
    recordingStartTime = null;
    recordingPausedTime = 0;
    recordingPausedAt = null;
    recordingCancelled = false;
}

async function uploadAudio(audioBlob) {
    try {
        showNotification(i18n.t('user.sending_audio'), 'info');
        
        const formData = new FormData();
        formData.append('file', audioBlob, 'audio.webm');
        formData.append('conversationId', currentConversationId);
        
        const response = await fetch(`${API_BASE}/api/chat/upload`, {
            method: 'POST',
            headers: {
                'Authorization': `Bearer ${localStorage.getItem('authToken')}`
            },
            body: formData
        });
        
        if (response.ok) {
            const data = await response.json();
            await sendMediaMessage(data.url, 'audio/webm');
            showNotification(i18n.t('user.audio_sent'), 'success');
        } else {
            throw new Error('Error fazer upload');
        }
    } catch (error) {
        console.error('Error enviar áudio:', error);
        showNotification(i18n.t('user.error_send_audio'), 'error');
    }
}

// ===== PAGAMENTO =====
window.openPaymentModal = function() {
    if (!currentConversation) {
        showNotification(i18n.t('user.select_conversation_first'), 'warning');
        return;
    }
    
    const modal = document.getElementById('paymentModal');
    modal.classList.add('active');
    
    // Preencher nome do cliente
    document.getElementById('paymentCustomerName').value = currentConversation.contact_name || '';
}

window.closePaymentModal = function() {
    const modal = document.getElementById('paymentModal');
    modal.classList.remove('active');
    
    // Limpar campos
    document.getElementById('paymentMethod').value = '';
    document.getElementById('paymentAmount').value = '';
    document.getElementById('paymentDescription').value = '';
    document.getElementById('paymentCustomerName').value = '';
}

window.createPaymentLink = async function() {
    if (!currentConversation) {
        showNotification(i18n.t('user.select_conversation_first'), 'warning');
        return;
    }
    
    const method = document.getElementById('paymentMethod').value;
    const amount = parseFloat(document.getElementById('paymentAmount').value);
    const description = document.getElementById('paymentDescription').value;
    const customerName = document.getElementById('paymentCustomerName').value;
    
    if (!method || !amount || amount <= 0) {
        showNotification(i18n.t('user.fill_required_fields'), 'warning');
        return;
    }
    
    if (!currentConversation.phone_number) {
        showNotification(i18n.t('user.customer_phone_not_found'), 'error');
        return;
    }
    
    try {
        const response = await authenticatedFetch(`${API_BASE}/api/payments/create-link`, {
            method: 'POST',
            body: JSON.stringify({
                payment_method: method,
                amount: amount,
                description: description || 'Pagamento',
                customer_phone: currentConversation.phone_number,
                customer_name: customerName || currentConversation.contact_name || 'Cliente'
            })
        });
        
        if (response.ok) {
            const data = await response.json();
            
            // Enviar link como mensagem
            const message = `💳 *Link de Pagamento*\n\n` +
                          `Valor: R$ ${amount.toFixed(2)}\n` +
                          `Descrição: ${description}\n` +
                          `Método: ${method.toUpperCase()}\n\n` +
                          `Link: ${data.payment_link.payment_url}`;
            
            await authenticatedFetch(`${API_BASE}/api/chat/conversations/${currentConversationId}/messages`, {
                method: 'POST',
                body: JSON.stringify({
                    message: message,
                    messageType: 'text'
                })
            });
            
            closePaymentModal();
            // REMOVIDO: await loadMessages(currentConversationId);
            // O socket já vai receber a nova mensagem e atualizar automaticamente
            showNotification(i18n.t('user.payment_link_created'), 'success');
        } else {
            const error = await response.json();
            throw new Error(error.error || 'Error criar link');
        }
    } catch (error) {
        console.error('Error criar link de pagamento:', error);
        showNotification(error.message || 'Error criar link de pagamento', 'error');
    }
}

// ===== UTILITÁRIOS =====
async function authenticatedFetch(url, options = {}) {
    const token = localStorage.getItem('authToken');
    
    if (!token) {
        console.error('Token not found no localStorage');
        showNotification(i18n.t('user.session_expired'), 'error');
        setTimeout(() => {
            window.location.href = '/';
        }, 2000);
        throw new Error('Token not found');
    }
    
    // Configuração padrão dos headers
    const defaultHeaders = {
        'Authorization': `Bearer ${token}`
    };
    
    // Se o body não for FormData, adicionar Content-Type
    if (!(options.body instanceof FormData)) {
        defaultHeaders['Content-Type'] = 'application/json';
    }
    
    // Merge dos headers
    const headers = {
        ...defaultHeaders,
        ...(options.headers || {})
    };
    
    // Debug
    console.log('=== authenticatedFetch DEBUG ===');
    console.log('URL:', url);
    console.log('Method:', options.method || 'GET');
    console.log('Token presente:', !!token);
    console.log('Token preview:', token ? token.substring(0, 20) + '...' : 'null');
    console.log('Headers:', headers);
    console.log('================================');
    
    try {
        const response = await fetch(url, {
            ...options,
            headers
        });
        
        // Se for 401, fazer logout automático
        if (response.status === 401) {
            console.error('Token inválido ou expirado (401)');
            localStorage.removeItem('authToken');
            localStorage.removeItem('currentUser');
            showNotification(i18n.t('user.session_expired'), 'error');
            setTimeout(() => {
                window.location.href = '/';
            }, 2000);
            throw new Error('Token inválido ou expirado');
        }
        
        return response;
    } catch (error) {
        console.error('Error in requisição:', error);
        throw error;
    }
}

function formatTime(timestamp) {
    if (!timestamp) return '';
    
    const date = new Date(timestamp);
    const now = new Date();
    const diff = now - date;
    
    // Menos de 24 horas: mostrar hora
    if (diff < 86400000) {
        return date.toLocaleTimeString('pt-BR', { hour: '2-digit', minute: '2-digit' });
    }
    
    // Menos de 7 dias: mostrar dia da semana
    if (diff < 604800000) {
        return date.toLocaleDateString('pt-BR', { weekday: 'short' });
    }
    
    // Mais de 7 dias: mostrar data
    return date.toLocaleDateString('pt-BR', { day: '2-digit', month: '2-digit' });
}

function showNotification(message, type = 'info') {
    const notification = document.getElementById('notification');
    notification.textContent = message;
    notification.className = `notification ${type} show`;
    
    setTimeout(() => {
        notification.classList.remove('show');
    }, 3000);
}

window.toggleEmojiPicker = function() {
    const picker = document.getElementById('emojiPicker');
    if (!picker) {
        createEmojiPicker();
    } else {
        picker.style.display = picker.style.display === 'none' ? 'block' : 'none';
    }
}

function createEmojiPicker() {
    const emojis = [
        '😀', '😃', '😄', '😁', '😆', '😅', '🤣', '😂', '🙂', '🙃',
        '😉', '😊', '😇', '🥰', '😍', '🤩', '😘', '😗', '😚', '😙',
        '😋', '😛', '😜', '🤪', '😝', '🤑', '🤗', '🤭', '🤫', '🤔',
        '🤐', '🤨', '😐', '😑', '😶', '😏', '😒', '🙄', '😬', '🤥',
        '😌', '😔', '😪', '🤤', '😴', '😷', '🤒', '🤕', '🤢', '🤮',
        '🤧', '🥵', '🥶', '😶‍🌫️', '🥴', '😵', '🤯', '🤠', '🥳', '😎',
        '🤓', '🧐', '😕', '😟', '🙁', '☹️', '😮', '😯', '😲', '😳',
        '🥺', '😦', '😧', '😨', '😰', '😥', '😢', '😭', '😱', '😖',
        '😣', '😞', '😓', '😩', '😫', '🥱', '😤', '😡', '😠', '🤬',
        '👍', '👎', '👌', '✌️', '🤞', '🤟', '🤘', '🤙', '👈', '👉',
        '👆', '👇', '☝️', '👏', '🙌', '👐', '🤲', '🤝', '🙏', '✍️',
        '❤️', '🧡', '💛', '💚', '💙', '💜', '🖤', '🤍', '🤎', '💔',
        '❣️', '💕', '💞', '💓', '💗', '💖', '💘', '💝', '💟', '☮️',
        '✝️', '☪️', '🕉️', '☸️', '✡️', '🔯', '🕎', '☯️', '☦️', '🛐'
    ];
    
    const picker = document.createElement('div');
    picker.id = 'emojiPicker';
    picker.className = 'emoji-picker';
    picker.innerHTML = emojis.map(emoji => 
        `<button class="emoji-btn" onclick="insertEmoji('${emoji}')">${emoji}</button>`
    ).join('');
    
    document.body.appendChild(picker);
}

window.insertEmoji = function(emoji) {
    const input = document.getElementById('messageInput');
    input.value += emoji;
    input.focus();
}

// Fechar emoji picker ao clicar fora
document.addEventListener('click', (e) => {
    const picker = document.getElementById('emojiPicker');
    const emojiBtn = e.target.closest('.emoji-btn-trigger');
    
    if (picker && !picker.contains(e.target) && !emojiBtn) {
        picker.style.display = 'none';
    }
});

window.logout = function() {
    localStorage.removeItem('authToken');
    localStorage.removeItem('currentUser');
    window.location.href = '/';
}

// ===== NOTIFICAÇÕES PUSH =====
if ('serviceWorker' in navigator && 'PushManager' in window) {
    navigator.serviceWorker.register('/sw.js').then(() => {
        console.log('Service Worker registrado');
    });
}

function requestNotificationPermission() {
    if ('Notification' in window && Notification.permission === 'default') {
        Notification.requestPermission();
    }
}

// ===== FUNÇÃO AUXILIAR PARA DURAÇÃO DO ÁUDIO =====
window.updateAudioDuration = function(audioElement, durationElementId) {
    const durationElement = document.getElementById(durationElementId);
    if (durationElement && audioElement.duration) {
        const minutes = Math.floor(audioElement.duration / 60);
        const seconds = Math.floor(audioElement.duration % 60);
        durationElement.textContent = `${minutes}:${seconds.toString().padStart(2, '0')}`;
    }
};

// ===== MENU DO CHAT =====
window.showChatMenu = function() {
    const chatMenuDropdown = document.getElementById('chatMenuDropdown');
    chatMenuDropdown.style.display = chatMenuDropdown.style.display === 'none' ? 'block' : 'none';
}

// ===== TRANSFERÊNCIA PARA LOJA =====
window.openTransferToStoreModal = async function() {
    try {
        // Fechar menu
        document.getElementById('chatMenuDropdown').style.display = 'none';
        
        // Carregar lista de lojas
        const response = await authenticatedFetch(`${API_BASE}/api/chat/stores`);
        const stores = await response.json();
        
        const storeSelect = document.getElementById('transferStore');
        storeSelect.innerHTML = `<option value="">${i18n.t('user.select_option')}</option>`;
        
        stores.forEach(store => {
            const option = document.createElement('option');
            option.value = store.id;
            option.textContent = store.name;
            storeSelect.appendChild(option);
        });
        
        document.getElementById('transferToStoreModal').style.display = 'block';
    } catch (error) {
        console.error('Error carregar lojas:', error);
        showNotification(i18n.t('user.error_load_stores'), 'error');
    }
}

window.closeTransferToStoreModal = function() {
    document.getElementById('transferToStoreModal').style.display = 'none';
}

window.confirmTransferToStore = async function() {
    try {
        const targetStore = document.getElementById('transferStore').value;
        
        if (!targetStore) {
            showNotification(i18n.t('user.select_store'), 'error');
            return;
        }
        
        if (!currentConversationId) {
            showNotification(i18n.t('user.no_conversation_selected'), 'error');
            return;
        }
        
        const response = await authenticatedFetch(`${API_BASE}/api/chat/conversations/${currentConversationId}/transfer`, {
            method: 'POST',
            body: JSON.stringify({ targetStore })
        });
        
        if (response.ok) {
            showNotification(i18n.t('user.conversation_transferred_success'), 'success');
            closeTransferToStoreModal();
            closeConversation();
        } else {
            const error = await response.json();
            showNotification(error.error || i18n.t('user.error_transfer_conversation'), 'error');
        }
    } catch (error) {
        console.error('Error transferir conversa:', error);
        showNotification(i18n.t('user.error_transfer_conversation'), 'error');
    }
}

// ===== TRANSFERÊNCIA PARA SUPORTE =====
window.openTransferToSupportModal = async function() {
    try {
        // Fechar menu
        document.getElementById('chatMenuDropdown').style.display = 'none';
        
        // Carregar lista de departamentos
        const response = await authenticatedFetch(`${API_BASE}/api/chat/departments`);
        const departments = await response.json();
        
        const departmentSelect = document.getElementById('transferDepartment');
        departmentSelect.innerHTML = `<option value="">${i18n.t('user.select_option')}</option>`;
        
        departments.forEach(dept => {
            const option = document.createElement('option');
            option.value = dept.name;
            option.textContent = dept.name;
            departmentSelect.appendChild(option);
        });
        
        document.getElementById('transferToSupportModal').style.display = 'block';
    } catch (error) {
        console.error('Error carregar departamentos:', error);
        showNotification(i18n.t('user.error_load_departments'), 'error');
    }
}

window.closeTransferToSupportModal = function() {
    document.getElementById('transferToSupportModal').style.display = 'none';
}

window.confirmTransferToSupport = async function() {
    try {
        const targetDepartment = document.getElementById('transferDepartment').value;
        
        if (!targetDepartment) {
            showNotification(i18n.t('user.select_department'), 'error');
            return;
        }
        
        if (!currentConversationId) {
            showNotification(i18n.t('user.no_conversation_selected'), 'error');
            return;
        }
        
        const response = await authenticatedFetch(`${API_BASE}/api/chat/conversations/${currentConversationId}/transfer-to-support`, {
            method: 'POST',
            body: JSON.stringify({ targetDepartment })
        });
        
        if (response.ok) {
            showNotification(i18n.t('user.conversation_transferred_support_success'), 'success');
            closeTransferToSupportModal();
            closeConversation();
        } else {
            const error = await response.json();
            showNotification(error.error || i18n.t('user.error_transfer_conversation'), 'error');
        }
    } catch (error) {
        console.error('Error transferir conversa:', error);
        showNotification(i18n.t('user.error_transfer_conversation'), 'error');
    }
}

// ===== ATUALIZAR FUNÇÃO DE ABRIR CONVERSA =====
const originalOpenConversation = window.openConversation;
window.openConversation = async function(conversationId) {
    await originalOpenConversation(conversationId);
    
    // Mostrar botão do menu do chat
    document.getElementById('chatMenuBtn').style.display = 'flex';
    document.getElementById('mainMenuBtn').style.display = 'none';
}

// ===== ATUALIZAR FUNÇÃO DE FECHAR CONVERSA =====
function closeConversationUpdated() {
    currentConversationId = null;
    currentConversation = null;
    
    // Atualizar header
    document.getElementById('headerTitle').textContent = 'Conversas';
    document.getElementById('headerSubtitle').textContent = '';
    document.getElementById('backBtn').style.display = 'none';
    
    // Mostrar/esconder botões corretos
    document.getElementById('chatMenuBtn').style.display = 'none';
    document.getElementById('mainMenuBtn').style.display = 'flex';
    
    // Mostrar botão de contatos quando voltar para lista de conversas
    const contactsBtn = document.querySelector('.floating-contacts-btn');
    if (contactsBtn) {
        contactsBtn.style.display = 'block';
    }
    
    // Trocar view
    document.getElementById('conversationsView').style.display = 'block';
    document.getElementById('chatView').style.display = 'none';
    
    // Recarregar conversas
    loadConversations();
}

// ===== FECHAR MENUS AO CLICAR FORA =====
document.addEventListener('click', (e) => {
    const chatMenuDropdown = document.getElementById('chatMenuDropdown');
    const mainMenuDropdown = document.getElementById('menuDropdown');
    
    if (chatMenuDropdown && !e.target.closest('#chatMenuBtn') && !e.target.closest('#chatMenuDropdown')) {
        chatMenuDropdown.style.display = 'none';
    }
    
    if (mainMenuDropdown && !e.target.closest('#mainMenuBtn') && !e.target.closest('#menuDropdown')) {
        mainMenuDropdown.style.display = 'none';
    }
});

// ===== CORRIGIR FUNÇÃO SHOWMENU =====
window.showMenu = function() {
    const menuDropdown = document.getElementById('menuDropdown');
    const chatMenuDropdown = document.getElementById('chatMenuDropdown');
    
    // Fechar chat menu se estiver aberto
    if (chatMenuDropdown) {
        chatMenuDropdown.style.display = 'none';
    }
    
    // Toggle main menu
    if (menuDropdown) {
        menuDropdown.style.display = menuDropdown.style.display === 'none' ? 'block' : 'none';
    }
}

// ===== CORRIGIR FUNÇÃO SHOWCHATMENU =====
window.showChatMenu = function() {
    const chatMenuDropdown = document.getElementById('chatMenuDropdown');
    const menuDropdown = document.getElementById('menuDropdown');
    
    // Fechar main menu se estiver aberto
    if (menuDropdown) {
        menuDropdown.style.display = 'none';
    }
    
    // Toggle chat menu
    if (chatMenuDropdown) {
        chatMenuDropdown.style.display = chatMenuDropdown.style.display === 'none' ? 'block' : 'none';
    }
}



// ===== PAINEL DE CONTATOS =====
let allContacts = [];

window.toggleContactsPanel = async function() {
    const panel = document.getElementById('contactsPanel');
    
    if (panel.style.display === 'none' || !panel.style.display) {
        panel.style.display = 'block';
        await loadContactsList();
    } else {
        panel.style.display = 'none';
        // Limpar busca ao fechar
        document.getElementById('contactsSearchInput').value = '';
    }
}

async function loadContactsList() {
    const listContainer = document.getElementById('contactsPanelList');
    
    try {
        listContainer.innerHTML = '<div class="contacts-loading">Loading contatos...</div>';
        
        const response = await authenticatedFetch(`${API_BASE}/api/contacts`);
        
        // Verificar se a resposta foi bem-sucedida
        if (!response.ok) {
            if (response.status === 403) {
                // Acesso negado - usuário não tem permissão
                listContainer.innerHTML = `
                    <div class="no-contacts">
                        <svg viewBox="0 0 24 24">
                            <path fill="currentColor" d="M12 1L3 5v6c0 5.55 3.84 10.74 9 12 5.16-1.26 9-6.45 9-12V5l-9-4zm0 10.99h7c-.53 4.12-3.28 7.79-7 8.94V12H5V6.3l7-3.11v8.8z"/>
                        </svg>
                        <p>Acesso negado. Apenas administradores podem visualizar contatos.</p>
                    </div>
                `;
                allContacts = [];
                return;
            }
            throw new Error(`Erro ${response.status}`);
        }
        
        const data = await response.json();
        
        // Verificar se data é um array
        if (Array.isArray(data)) {
            allContacts = data;
        } else {
            allContacts = [];
        }
        
        renderContactsList(allContacts);
    } catch (error) {
        console.error('Error carregar contatos:', error);
        listContainer.innerHTML = `
            <div class="no-contacts">
                <svg viewBox="0 0 24 24">
                    <path fill="currentColor" d="M20 6h-8l-2-2H4c-1.1 0-1.99.9-1.99 2L2 18c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V8c0-1.1-.9-2-2-2z"/>
                </svg>
                <p>Error carregar contatos</p>
            </div>
        `;
        allContacts = [];
    }
}

function renderContactsList(contacts) {
    const listContainer = document.getElementById('contactsPanelList');
    
    if (!contacts || contacts.length === 0) {
        listContainer.innerHTML = `
            <div class="no-contacts">
                <svg viewBox="0 0 24 24">
                    <path fill="currentColor" d="M20 6h-8l-2-2H4c-1.1 0-1.99.9-1.99 2L2 18c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V8c0-1.1-.9-2-2-2z"/>
                </svg>
                <p>Nenhum contato cadastrado</p>
            </div>
        `;
        return;
    }
    
    // Ordenar alfabeticamente por nome
    const sortedContacts = [...contacts].sort((a, b) => {
        const nameA = (a.name || '').toLowerCase();
        const nameB = (b.name || '').toLowerCase();
        return nameA.localeCompare(nameB);
    });
    
    listContainer.innerHTML = sortedContacts.map(contact => {
        const initials = (contact.name || contact.phone).substring(0, 2).toUpperCase();
        
        return `
            <div class="contact-item" onclick="openContactConversation('${contact.phone}', '${contact.name}')">
                <div class="contact-avatar">${initials}</div>
                <div class="contact-info">
                    <div class="contact-name">${contact.name || contact.phone}</div>
                    <div class="contact-phone">${contact.phone}</div>
                </div>
            </div>
        `;
    }).join('');
}

window.filterContactsList = function() {
    const searchInput = document.getElementById('contactsSearchInput');
    const query = searchInput.value.toLowerCase().trim();
    
    if (!query) {
        renderContactsList(allContacts);
        return;
    }
    
    const filtered = allContacts.filter(contact => {
        const name = (contact.name || '').toLowerCase();
        const phone = (contact.phone || '').toLowerCase();
        return name.includes(query) || phone.includes(query);
    });
    
    renderContactsList(filtered);
}

window.openContactConversation = async function(phone, name) {
    try {
        console.log('=== Tentando abrir conversa ===');
        console.log('Phone:', phone);
        console.log('Name:', name);
        console.log('Token:', localStorage.getItem('authToken') ? 'Presente' : 'Ausente');
        
        // Fechar o painel de contatos
        toggleContactsPanel();
        
        // Search se já existe uma conversa com este contato
        const response = await authenticatedFetch(`${API_BASE}/api/chat/conversations`);
        
        if (!response.ok) {
            const error = await response.json();
            throw new Error(error.error || 'Error buscar conversas');
        }
        
        const conversations = await response.json();
        
        const existingConv = conversations.find(conv => conv.phone_number === phone);
        
        if (existingConv) {
            // Se já existe, abrir a conversa
            console.log('Conversa existente encontrada:', existingConv.id);
            openConversation(existingConv.id);
        } else {
            // Se não existe, criar uma nova conversa
            console.log('Criando nova conversa...');
            const createResponse = await authenticatedFetch(`${API_BASE}/api/chat/conversations`, {
                method: 'POST',
                body: JSON.stringify({
                    phone_number: phone,
                    contact_name: name
                })
            });
            
            if (!createResponse.ok) {
                const error = await createResponse.json();
                throw new Error(error.error || 'Error criar conversa');
            }
            
            const newConv = await createResponse.json();
            console.log('Nova conversa criada:', newConv);
            
            // Recarregar conversas e abrir a nova
            await loadConversations();
            openConversation(newConv.id);
            showNotification(`Conversa iniciada com ${name}`, 'success');
        }
    } catch (error) {
        console.error('Error abrir conversa do contato:', error);
        showNotification(error.message || i18n.t('user.error_open_conversation'), 'error');
    }
}

