// Translations Admin JavaScript

let languages = [];
let translations = [];
let currentLanguage = '';
let selectedFile = null;
let unsavedChanges = new Set();

// Initialization
document.addEventListener('DOMContentLoaded', function() {
    loadLanguages();
    loadActiveLanguage();
    
    // Event listeners
    document.getElementById('translationSearch').addEventListener('input', debounce(searchTranslations, 500));
    
    // Drag and drop
    setupDragAndDrop();
});

// ===== TAB NAVIGATION =====

function showTab(tabName) {
    // Hide all tabs
    document.querySelectorAll('.tab-content').forEach(tab => {
        tab.classList.remove('active');
    });
    
    // Remove active class from all buttons
    document.querySelectorAll('.tab').forEach(btn => {
        btn.classList.remove('active');
    });
    
    // Show selected tab
    document.getElementById(tabName).classList.add('active');
    
    // Activate corresponding button
    event.target.classList.add('active');
    
    // Load tab-specific data
    switch(tabName) {
        case 'languages':
            loadLanguages();
            break;
        case 'translate':
            populateLanguageSelects();
            break;
        case 'import':
            populateLanguageSelects();
            break;
    }
}

// ===== LANGUAGES =====

async function loadLanguages() {
    try {
        const token = localStorage.getItem('token');
        const response = await fetch('/api/translations/languages', {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            languages = await response.json();
            renderLanguages();
            populateLanguageSelects();
        }
    } catch (error) {
        console.error('Error loading languages:', error);
        showNotification('Error loading languages', 'error');
    }
}

async function loadActiveLanguage() {
    try {
        const token = localStorage.getItem('token');
        const response = await fetch('/api/translations/active', {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            const data = await response.json();
            window.activeLanguage = data.active_language;
        }
    } catch (error) {
        console.error('Error loading active language:', error);
    }
}

function renderLanguages() {
    const container = document.getElementById('languagesList');
    container.innerHTML = '';
    
    if (languages.length === 0) {
        container.innerHTML = '<p>No language configured. Create the first language above.</p>';
        return;
    }
    
    languages.forEach(language => {
        const languageDiv = document.createElement('div');
        languageDiv.className = 'language-card';
        
        if (language.language_code === window.activeLanguage) {
            languageDiv.classList.add('active');
        }
        
        // Determinar status da tradução
        const completionRate = language.translation_count > 0 ? 
            Math.round((getTranslatedCount(language) / language.translation_count) * 100) : 0;
        
        let statusClass = 'status-empty';
        if (completionRate > 80) statusClass = 'status-complete';
        else if (completionRate > 20) statusClass = 'status-partial';
        
        languageDiv.innerHTML = `
            <div class="language-info">
                <div class="language-name">
                    <span class="status-indicator ${statusClass}"></span>
                    ${language.language_name}
                    ${language.language_code === window.activeLanguage ? '<span class="active-badge">ATIVO</span>' : ''}
                </div>
                <div class="language-code">${language.language_code}</div>
                <div class="language-stats">
                    ${language.translation_count} strings • ${completionRate}% traduzido
                    ${language.last_updated ? `• Atualizado: ${new Date(language.last_updated).toLocaleDateString('pt-BR')}` : ''}
                </div>
            </div>
            <div class="language-actions">
                <button class="btn btn-info" onclick="editLanguage('${language.language_code}')">✏️ Traduzir</button>
                <button class="btn btn-warning" onclick="downloadLanguageFile('${language.language_code}')">📥 Baixar</button>
                ${language.language_code !== window.activeLanguage ? 
                    `<button class="btn btn-success" onclick="applyLanguage('${language.language_code}')">✅ Aplicar</button>` : 
                    '<span style="color: #4CAF50; font-weight: bold;">Em Uso</span>'
                }
            </div>
        `;
        
        container.appendChild(languageDiv);
    });
}

function getTranslatedCount(language) {
    // Simular contagem de strings traduzidas
    // Na implementação real, isso viria do backend
    return Math.floor(language.translation_count * 0.7);
}

async function createLanguage() {
    const code = document.getElementById('languageCode').value.trim();
    const name = document.getElementById('languageName').value.trim();
    
    if (!code || !name) {
        showNotification('Código e nome do idioma são obrigatórios', 'error');
        return;
    }
    
    try {
        const token = localStorage.getItem('token');
        const response = await fetch('/api/translations/languages', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({
                language_code: code,
                language_name: name
            })
        });
        
        if (response.ok) {
            const result = await response.json();
            showNotification(result.message, 'success');
            
            // Limpar formulário
            document.getElementById('languageCode').value = '';
            document.getElementById('languageName').value = '';
            
            loadLanguages();
        } else {
            const error = await response.json();
            throw new Error(error.error || 'Erro ao criar idioma');
        }
    } catch (error) {
        console.error('Erro ao criar idioma:', error);
        showNotification(error.message, 'error');
    }
}

function editLanguage(languageCode) {
    // Mudar para aba de tradução
    showTab('translate');
    
    // Selecionar idioma
    document.getElementById('translateLanguageSelect').value = languageCode;
    loadTranslations();
}

async function applyLanguage(languageCode) {
    if (!confirm(`Tem certeza que deseja aplicar o idioma ${languageCode} como ativo no sistema?`)) {
        return;
    }
    
    try {
        const token = localStorage.getItem('token');
        const response = await fetch(`/api/translations/apply/${languageCode}`, {
            method: 'POST',
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            const result = await response.json();
            showNotification(result.message, 'success');
            
            window.activeLanguage = languageCode;
            loadLanguages();
        } else {
            const error = await response.json();
            throw new Error(error.error || 'Erro ao aplicar idioma');
        }
    } catch (error) {
        console.error('Erro ao aplicar idioma:', error);
        showNotification(error.message, 'error');
    }
}

// ===== TRADUÇÕES =====

function populateLanguageSelects() {
    const selects = [
        'translateLanguageSelect',
        'downloadLanguageSelect'
    ];
    
    selects.forEach(selectId => {
        const select = document.getElementById(selectId);
        if (select) {
            // Manter opção padrão
            const defaultOption = select.querySelector('option[value=""]');
            select.innerHTML = '';
            if (defaultOption) {
                select.appendChild(defaultOption);
            }
            
            languages.forEach(language => {
                const option = document.createElement('option');
                option.value = language.language_code;
                option.textContent = `${language.language_name} (${language.language_code})`;
                select.appendChild(option);
            });
        }
    });
}

async function loadTranslations(page = 1) {
    const languageCode = document.getElementById('translateLanguageSelect').value;
    
    if (!languageCode) {
        document.getElementById('translationsContainer').innerHTML = 
            '<p>Selecione um idioma para começar a traduzir.</p>';
        return;
    }
    
    try {
        const token = localStorage.getItem('token');
        const search = document.getElementById('translationSearch').value;
        
        const params = new URLSearchParams({
            page: page,
            limit: 20,
            search: search
        });
        
        const response = await fetch(`/api/translations/strings/${languageCode}?${params}`, {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            const data = await response.json();
            translations = data.translations;
            currentLanguage = languageCode;
            renderTranslations(data);
        }
    } catch (error) {
        console.error('Erro ao carregar traduções:', error);
        showNotification('Erro ao carregar traduções', 'error');
    }
}

function renderTranslations(data) {
    const container = document.getElementById('translationsContainer');
    
    if (data.translations.length === 0) {
        container.innerHTML = '<p>Nenhuma string encontrada para este idioma.</p>';
        return;
    }
    
    let html = `
        <table class="translations-table">
            <thead>
                <tr>
                    <th style="width: 30%;">Chave</th>
                    <th style="width: 60%;">Tradução</th>
                    <th style="width: 10%;">Status</th>
                </tr>
            </thead>
            <tbody>
    `;
    
    data.translations.forEach((translation, index) => {
        const isEmpty = !translation.translation_value || translation.translation_value.trim() === '';
        const statusIcon = isEmpty ? '❌' : '✅';
        const statusTitle = isEmpty ? 'Não traduzido' : 'Traduzido';
        
        html += `
            <tr>
                <td>
                    <div class="translation-key">${translation.translation_key}</div>
                </td>
                <td class="translation-value">
                    <textarea 
                        class="translation-input" 
                        data-key="${translation.translation_key}"
                        placeholder="Digite a tradução..."
                        onchange="markAsChanged(this)"
                        onblur="autoSaveTranslation(this)"
                    >${translation.translation_value || ''}</textarea>
                    <span class="save-indicator" id="save-${index}" style="display: none;">💾 Salvo</span>
                </td>
                <td style="text-align: center;">
                    <span title="${statusTitle}">${statusIcon}</span>
                </td>
            </tr>
        `;
    });
    
    html += '</tbody></table>';
    container.innerHTML = html;
    
    // Renderizar paginação
    renderPagination('translationsPagination', data, loadTranslations);
}

function markAsChanged(input) {
    const key = input.dataset.key;
    unsavedChanges.add(key);
    
    // Mostrar indicador visual
    input.style.borderColor = '#FF9800';
}

async function autoSaveTranslation(input) {
    const key = input.dataset.key;
    const value = input.value;
    
    try {
        const token = localStorage.getItem('token');
        const response = await fetch(`/api/translations/strings/${currentLanguage}/${encodeURIComponent(key)}`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({
                translation_value: value
            })
        });
        
        if (response.ok) {
            // Remover da lista de mudanças não salvas
            unsavedChanges.delete(key);
            
            // Mostrar indicador de salvo
            input.style.borderColor = '#4CAF50';
            
            // Mostrar mensagem temporária
            const saveIndicator = input.parentElement.querySelector('.save-indicator');
            if (saveIndicator) {
                saveIndicator.style.display = 'inline';
                setTimeout(() => {
                    saveIndicator.style.display = 'none';
                }, 2000);
            }
        } else {
            throw new Error('Erro ao salvar tradução');
        }
    } catch (error) {
        console.error('Erro ao salvar tradução:', error);
        input.style.borderColor = '#f44336';
        showNotification('Erro ao salvar tradução', 'error');
    }
}

async function saveAllTranslations() {
    if (unsavedChanges.size === 0) {
        showNotification('Não há alterações para salvar', 'info');
        return;
    }
    
    const inputs = document.querySelectorAll('.translation-input');
    let saved = 0;
    let errors = 0;
    
    for (const input of inputs) {
        const key = input.dataset.key;
        if (unsavedChanges.has(key)) {
            try {
                await autoSaveTranslation(input);
                saved++;
            } catch (error) {
                errors++;
            }
        }
    }
    
    showNotification(`${saved} traduções salvas${errors > 0 ? `, ${errors} erros` : ''}`, 
                    errors > 0 ? 'warning' : 'success');
}

function searchTranslations() {
    loadTranslations(1);
}

// ===== IMPORTAR/EXPORTAR =====

async function downloadBaseFile() {
    try {
        const token = localStorage.getItem('token');
        const response = await fetch('/api/translations/download/base', {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            const blob = await response.blob();
            downloadBlob(blob, 'en.json');
            showNotification('Arquivo base baixado com sucesso!', 'success');
        } else {
            throw new Error('Erro ao baixar arquivo base');
        }
    } catch (error) {
        console.error('Erro ao baixar arquivo base:', error);
        showNotification('Erro ao baixar arquivo base', 'error');
    }
}

async function downloadLanguageFile(languageCode) {
    if (!languageCode) {
        languageCode = document.getElementById('downloadLanguageSelect').value;
    }
    
    if (!languageCode) {
        showNotification('Selecione um idioma para baixar', 'error');
        return;
    }
    
    try {
        const token = localStorage.getItem('token');
        const response = await fetch(`/api/translations/download/${languageCode}`, {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            const blob = await response.blob();
            downloadBlob(blob, `${languageCode}.json`);
            showNotification(`Arquivo ${languageCode}.json baixado com sucesso!`, 'success');
        } else {
            throw new Error('Erro ao baixar arquivo de tradução');
        }
    } catch (error) {
        console.error('Erro ao baixar tradução:', error);
        showNotification('Erro ao baixar arquivo de tradução', 'error');
    }
}

function downloadBlob(blob, filename) {
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.style.display = 'none';
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    window.URL.revokeObjectURL(url);
    document.body.removeChild(a);
}

// ===== UPLOAD DE ARQUIVOS =====

function setupDragAndDrop() {
    const uploadArea = document.querySelector('.upload-area');
    
    uploadArea.addEventListener('dragover', handleDragOver);
    uploadArea.addEventListener('dragleave', handleDragLeave);
    uploadArea.addEventListener('drop', handleFileDrop);
}

function handleDragOver(e) {
    e.preventDefault();
    e.currentTarget.classList.add('dragover');
}

function handleDragLeave(e) {
    e.preventDefault();
    e.currentTarget.classList.remove('dragover');
}

function handleFileDrop(e) {
    e.preventDefault();
    e.currentTarget.classList.remove('dragover');
    
    const files = e.dataTransfer.files;
    if (files.length > 0) {
        handleFileSelection(files[0]);
    }
}

function handleFileSelect(e) {
    const file = e.target.files[0];
    if (file) {
        handleFileSelection(file);
    }
}

function handleFileSelection(file) {
    // Validar tipo de arquivo
    if (!file.name.endsWith('.json')) {
        showNotification('Apenas arquivos JSON são permitidos', 'error');
        return;
    }
    
    // Validar nome do arquivo (deve ser código de idioma)
    const languageCode = file.name.replace('.json', '');
    if (!/^[a-z]{2}(-[A-Z]{2})?$/.test(languageCode)) {
        showNotification('Nome do arquivo deve seguir o padrão do código do idioma (ex: pt.json, en.json)', 'error');
        return;
    }
    
    selectedFile = file;
    
    // Mostrar informações do arquivo
    const fileInfo = document.getElementById('fileInfo');
    fileInfo.innerHTML = `
        <h4>📁 Arquivo Selecionado</h4>
        <p><strong>Nome:</strong> ${file.name}</p>
        <p><strong>Tamanho:</strong> ${(file.size / 1024).toFixed(2)} KB</p>
        <p><strong>Idioma Detectado:</strong> ${languageCode}</p>
        <p><strong>Tipo:</strong> ${file.type || 'application/json'}</p>
    `;
    fileInfo.style.display = 'block';
    
    // Habilitar botão de upload
    document.getElementById('uploadBtn').disabled = false;
}

async function uploadTranslation() {
    if (!selectedFile) {
        showNotification('Selecione um arquivo primeiro', 'error');
        return;
    }
    
    const formData = new FormData();
    formData.append('translationFile', selectedFile);
    
    // Mostrar progresso
    const progressContainer = document.getElementById('uploadProgress');
    const progressFill = document.getElementById('progressFill');
    const progressText = document.getElementById('progressText');
    
    progressContainer.style.display = 'block';
    progressFill.style.width = '0%';
    progressText.textContent = 'Iniciando upload...';
    
    try {
        const token = localStorage.getItem('token');
        
        // Simular progresso
        let progress = 0;
        const progressInterval = setInterval(() => {
            progress += 10;
            progressFill.style.width = progress + '%';
            
            if (progress < 50) {
                progressText.textContent = 'Enviando arquivo...';
            } else if (progress < 90) {
                progressText.textContent = 'Processando traduções...';
            } else {
                progressText.textContent = 'Finalizando...';
            }
        }, 200);
        
        const response = await fetch('/api/translations/upload', {
            method: 'POST',
            headers: {
                'Authorization': `Bearer ${token}`
            },
            body: formData
        });
        
        clearInterval(progressInterval);
        progressFill.style.width = '100%';
        
        if (response.ok) {
            const result = await response.json();
            progressText.textContent = 'Upload concluído!';
            
            showNotification(
                `Tradução importada! ${result.imported} novas, ${result.updated} atualizadas. ${result.totalErrors} erros.`,
                'success'
            );
            
            // Limpar formulário
            selectedFile = null;
            document.getElementById('translationFile').value = '';
            document.getElementById('fileInfo').style.display = 'none';
            document.getElementById('uploadBtn').disabled = true;
            
            // Recarregar dados
            loadLanguages();
            
            setTimeout(() => {
                progressContainer.style.display = 'none';
            }, 3000);
        } else {
            const error = await response.json();
            throw new Error(error.error || 'Erro ao importar tradução');
        }
    } catch (error) {
        console.error('Erro ao fazer upload:', error);
        progressText.textContent = 'Erro no upload';
        progressFill.style.backgroundColor = '#f44336';
        showNotification(error.message, 'error');
        
        setTimeout(() => {
            progressContainer.style.display = 'none';
            progressFill.style.backgroundColor = '#4CAF50';
        }, 3000);
    }
}

// ===== UTILITÁRIOS =====

function renderPagination(containerId, data, loadFunction) {
    const container = document.getElementById(containerId);
    if (!container) return;
    
    const { page, totalPages } = data;
    let html = '';
    
    // Botão anterior
    if (page > 1) {
        html += `<button onclick="${loadFunction.name}(${page - 1})">« Anterior</button>`;
    }
    
    // Números das páginas
    const startPage = Math.max(1, page - 2);
    const endPage = Math.min(totalPages, page + 2);
    
    for (let i = startPage; i <= endPage; i++) {
        html += `<button class="${i === page ? 'active' : ''}" onclick="${loadFunction.name}(${i})">${i}</button>`;
    }
    
    // Botão próximo
    if (page < totalPages) {
        html += `<button onclick="${loadFunction.name}(${page + 1})">Próximo »</button>`;
    }
    
    container.innerHTML = html;
}

function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

function showNotification(message, type = 'info') {
    // Criar elemento de notificação
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px 20px;
        border-radius: 5px;
        color: white;
        font-weight: bold;
        z-index: 10000;
        animation: slideIn 0.3s ease;
        max-width: 400px;
    `;
    
    // Cores por tipo
    const colors = {
        success: '#4CAF50',
        error: '#f44336',
        info: '#2196F3',
        warning: '#FF9800'
    };
    
    notification.style.backgroundColor = colors[type] || colors.info;
    
    document.body.appendChild(notification);
    
    // Remover após 5 segundos
    setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 5000);
}

// CSS para animações
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
`;
document.head.appendChild(style);
