// PWA and Push Notifications JavaScript

class PWAManager {
    constructor() {
        this.swRegistration = null;
        this.isSubscribed = false;
        this.applicationServerKey = null;
        
        this.init();
    }
    
    async init() {
        // Check Service Worker support
        if ('serviceWorker' in navigator) {
            await this.registerServiceWorker();
        }
        
        // Check Push Notifications support
        if ('PushManager' in window) {
            await this.initializePushNotifications();
        }
        
        // Setup PWA installation events
        this.setupInstallPrompt();
        
        // Setup connectivity events
        this.setupConnectivityEvents();
        
        // Setup background sync
        this.setupBackgroundSync();
    }
    
    // ===== SERVICE WORKER =====
    
    async registerServiceWorker() {
        try {
            this.swRegistration = await navigator.serviceWorker.register('/sw.js');
            console.log('[PWA] Service Worker registered:', this.swRegistration);
            
            // Listen to Service Worker updates
            this.swRegistration.addEventListener('updatefound', () => {
                const newWorker = this.swRegistration.installing;
                
                newWorker.addEventListener('statechange', () => {
                    if (newWorker.state === 'installed' && navigator.serviceWorker.controller) {
                        this.showUpdateAvailable();
                    }
                });
            });
            
            // Listen to Service Worker messages
            navigator.serviceWorker.addEventListener('message', (event) => {
                this.handleServiceWorkerMessage(event);
            });
            
        } catch (error) {
            console.error('[PWA] Error registering Service Worker:', error);
        }
    }
    
    handleServiceWorkerMessage(event) {
        const { type, data } = event.data || {};
        
        switch (type) {
            case 'CACHE_UPDATED':
                console.log('[PWA] Cache updated');
                break;
            case 'OFFLINE_READY':
                this.showOfflineReady();
                break;
            case 'UPDATE_AVAILABLE':
                this.showUpdateAvailable();
                break;
        }
    }
    
    // ===== PUSH NOTIFICATIONS =====
    
    async initializePushNotifications() {
        try {
            // Fetch VAPID public key
            const response = await fetch('/api/push-notifications/vapid-public-key');
            
            if (!response.ok) {
                console.warn('[PWA] Push notifications endpoint not available');
                return;
            }
            
            const data = await response.json();
            
            if (!data.publicKey) {
                console.warn('[PWA] VAPID public key not found');
                return;
            }
            
            this.applicationServerKey = this.urlBase64ToUint8Array(data.publicKey);
            
            // Check subscription status
            await this.checkSubscriptionStatus();
            
        } catch (error) {
            console.error('[PWA] Error initializing push notifications:', error);
        }
    }
    
    async checkSubscriptionStatus() {
        if (!this.swRegistration) return;
        
        try {
            const subscription = await this.swRegistration.pushManager.getSubscription();
            this.isSubscribed = subscription !== null;
            
            if (this.isSubscribed) {
                console.log('[PWA] Push notifications already enabled');
                this.updatePushButton(true);
            } else {
                console.log('[PWA] Push notifications not enabled');
                this.updatePushButton(false);
            }
            
        } catch (error) {
            console.error('[PWA] Error checking subscription:', error);
        }
    }
    
    async subscribeToPush() {
        if (!this.swRegistration || !this.applicationServerKey) {
            console.error('[PWA] Service Worker or VAPID key not available');
            return false;
        }
        
        try {
            // Request permission
            const permission = await Notification.requestPermission();
            
            if (permission !== 'granted') {
                console.log('[PWA] Notification permission denied');
                this.showNotificationPermissionDenied();
                return false;
            }
            
            // Create subscription
            const subscription = await this.swRegistration.pushManager.subscribe({
                userVisibleOnly: true,
                applicationServerKey: this.applicationServerKey
            });
            
            // Send subscription to server
            const success = await this.sendSubscriptionToServer(subscription);
            
            if (success) {
                this.isSubscribed = true;
                this.updatePushButton(true);
                this.showNotificationSuccess('Push notifications enabled successfully!');
                return true;
            }
            
        } catch (error) {
            console.error('[PWA] Error enabling push notifications:', error);
            this.showNotificationError('Error enabling push notifications');
            return false;
        }
    }
    
    async unsubscribeFromPush() {
        if (!this.swRegistration) return false;
        
        try {
            const subscription = await this.swRegistration.pushManager.getSubscription();
            
            if (subscription) {
                // Cancel on server
                await this.removeSubscriptionFromServer(subscription);
                
                // Cancel locally
                await subscription.unsubscribe();
                
                this.isSubscribed = false;
                this.updatePushButton(false);
                this.showNotificationSuccess('Push notifications disabled');
                return true;
            }
            
        } catch (error) {
            console.error('[PWA] Error disabling push notifications:', error);
            this.showNotificationError('Error disabling push notifications');
            return false;
        }
    }
    
    async sendSubscriptionToServer(subscription) {
        try {
            const token = localStorage.getItem('token');
            const deviceInfo = this.getDeviceInfo();
            
            const response = await fetch('/api/push-notifications/subscribe', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify({
                    subscription,
                    deviceInfo
                })
            });
            
            return response.ok;
        } catch (error) {
            console.error('[PWA] Error sending subscription:', error);
            return false;
        }
    }
    
    async removeSubscriptionFromServer(subscription) {
        try {
            const token = localStorage.getItem('token');
            
            const response = await fetch('/api/push-notifications/unsubscribe', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify({
                    endpoint: subscription.endpoint
                })
            });
            
            return response.ok;
        } catch (error) {
            console.error('[PWA] Error removing subscription:', error);
            return false;
        }
    }
    
    // ===== PWA INSTALLATION =====
    
    setupInstallPrompt() {
        let deferredPrompt = null;
        
        // Listen to installation event
        window.addEventListener('beforeinstallprompt', (e) => {
            console.log('[PWA] Installation prompt available');
            e.preventDefault();
            deferredPrompt = e;
            this.showInstallButton();
        });
        
        // Listen to successful installation
        window.addEventListener('appinstalled', () => {
            console.log('[PWA] App installed successfully');
            this.hideInstallButton();
            this.showNotificationSuccess('App installed successfully!');
            deferredPrompt = null;
        });
        
        // Setup install button
        const installButton = document.getElementById('installButton');
        if (installButton) {
            installButton.addEventListener('click', async () => {
                if (deferredPrompt) {
                    deferredPrompt.prompt();
                    const { outcome } = await deferredPrompt.userChoice;
                    
                    if (outcome === 'accepted') {
                        console.log('[PWA] User accepted installation');
                    } else {
                        console.log('[PWA] User declined installation');
                    }
                    
                    deferredPrompt = null;
                }
            });
        }
    }
    
    // ===== CONNECTIVITY =====
    
    setupConnectivityEvents() {
        // Connection events
        window.addEventListener('online', () => {
            console.log('[PWA] Connection restored');
            this.showConnectivityStatus(true);
            this.syncPendingData();
        });
        
        window.addEventListener('offline', () => {
            console.log('[PWA] Connection lost');
            this.showConnectivityStatus(false);
        });
        
        // Check initial status
        this.showConnectivityStatus(navigator.onLine);
    }
    
    showConnectivityStatus(isOnline) {
        const statusElement = document.getElementById('connectivityStatus');
        if (statusElement) {
            statusElement.className = `connectivity-status ${isOnline ? 'online' : 'offline'}`;
            statusElement.textContent = isOnline ? 'Online' : 'Offline';
        }
        
        // Show temporary notification
        if (isOnline) {
            this.showNotificationSuccess('Connection restored');
        } else {
            this.showNotificationWarning('You are offline. Some features may be limited.');
        }
    }
    
    // ===== BACKGROUND SYNC =====
    
    setupBackgroundSync() {
        if ('serviceWorker' in navigator && 'sync' in window.ServiceWorkerRegistration.prototype) {
            console.log('[PWA] Background sync available');
        }
    }
    
    async syncPendingData() {
        if (this.swRegistration && 'sync' in window.ServiceWorkerRegistration.prototype) {
            try {
                await this.swRegistration.sync.register('sync-messages');
                await this.swRegistration.sync.register('sync-contacts');
                console.log('[PWA] Background sync scheduled');
            } catch (error) {
                console.error('[PWA] Error scheduling sync:', error);
            }
        }
    }
    
    // ===== INTERFACE =====
    
    updatePushButton(isSubscribed) {
        const pushButton = document.getElementById('pushButton');
        if (pushButton) {
            if (isSubscribed) {
                pushButton.textContent = '🔕 Disable Notifications';
                pushButton.onclick = () => this.unsubscribeFromPush();
            } else {
                pushButton.textContent = '🔔 Enable Notifications';
                pushButton.onclick = () => this.subscribeToPush();
            }
        }
    }
    
    showInstallButton() {
        const installButton = document.getElementById('installButton');
        if (installButton) {
            installButton.style.display = 'block';
        }
    }
    
    hideInstallButton() {
        const installButton = document.getElementById('installButton');
        if (installButton) {
            installButton.style.display = 'none';
        }
    }
    
    showUpdateAvailable() {
        const updateBanner = document.getElementById('updateBanner');
        if (updateBanner) {
            updateBanner.style.display = 'block';
        } else {
            this.showNotificationInfo('New version available! Reload the page to update.');
        }
    }
    
    showOfflineReady() {
        this.showNotificationSuccess('App ready for offline use!');
    }
    
    showNotificationPermissionDenied() {
        this.showNotificationWarning('Notification permission denied. You can enable it in browser settings.');
    }
    
    // ===== INTERFACE NOTIFICATIONS =====
    
    showNotificationSuccess(message) {
        this.showNotification(message, 'success');
    }
    
    showNotificationError(message) {
        this.showNotification(message, 'error');
    }
    
    showNotificationWarning(message) {
        this.showNotification(message, 'warning');
    }
    
    showNotificationInfo(message) {
        this.showNotification(message, 'info');
    }
    
    showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `pwa-notification ${type}`;
        notification.textContent = message;
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 15px 20px;
            border-radius: 8px;
            color: white;
            font-weight: bold;
            z-index: 10000;
            animation: slideIn 0.3s ease;
            max-width: 350px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        `;
        
        // Colors by type
        const colors = {
            success: '#4CAF50',
            error: '#f44336',
            info: '#2196F3',
            warning: '#FF9800'
        };
        
        notification.style.backgroundColor = colors[type] || colors.info;
        
        document.body.appendChild(notification);
        
        // Remove after 5 seconds
        setTimeout(() => {
            notification.style.animation = 'slideOut 0.3s ease';
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }, 5000);
    }
    
    // ===== UTILITIES =====
    
    urlBase64ToUint8Array(base64String) {
        if (!base64String) {
            console.error('[PWA] Base64 string is undefined or null');
            return new Uint8Array(0);
        }
        
        const padding = '='.repeat((4 - base64String.length % 4) % 4);
        const base64 = (base64String + padding)
            .replace(/-/g, '+')
            .replace(/_/g, '/');
        
        const rawData = window.atob(base64);
        const outputArray = new Uint8Array(rawData.length);
        
        for (let i = 0; i < rawData.length; ++i) {
            outputArray[i] = rawData.charCodeAt(i);
        }
        
        return outputArray;
    }
    
    getDeviceInfo() {
        return {
            userAgent: navigator.userAgent,
            platform: navigator.platform,
            language: navigator.language,
            screen: {
                width: screen.width,
                height: screen.height
            },
            viewport: {
                width: window.innerWidth,
                height: window.innerHeight
            },
            timestamp: new Date().toISOString()
        };
    }
    
    // ===== PUBLIC METHODS =====
    
    async clearCache() {
        if (this.swRegistration) {
            this.swRegistration.postMessage({ type: 'CLEAR_CACHE' });
            this.showNotificationSuccess('Cache cleared successfully');
        }
    }
    
    async updateApp() {
        if (this.swRegistration && this.swRegistration.waiting) {
            this.swRegistration.waiting.postMessage({ type: 'SKIP_WAITING' });
            window.location.reload();
        }
    }
    
    getInstallationStatus() {
        return {
            isInstalled: window.matchMedia('(display-mode: standalone)').matches,
            isInstallable: !!window.deferredPrompt,
            isPushEnabled: this.isSubscribed,
            isServiceWorkerActive: !!this.swRegistration
        };
    }
}

// ===== INICIALIZAÇÃO =====

let pwaManager;

document.addEventListener('DOMContentLoaded', () => {
    pwaManager = new PWAManager();
    
    // Expose globally for use in other parts of the application
    window.pwaManager = pwaManager;
});

// ===== CSS FOR ANIMATIONS =====

const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
    
    .connectivity-status {
        padding: 5px 10px;
        border-radius: 15px;
        font-size: 12px;
        font-weight: bold;
        text-transform: uppercase;
    }
    
    .connectivity-status.online {
        background: #d4edda;
        color: #155724;
    }
    
    .connectivity-status.offline {
        background: #f8d7da;
        color: #721c24;
    }
    
    .pwa-notification {
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    }
`;
document.head.appendChild(style);

// ===== EXPORT FOR USE IN OTHER MODULES =====

if (typeof module !== 'undefined' && module.exports) {
    module.exports = PWAManager;
}
