// ===================================================================
//  EDIT PROFILE - COLOR AND SYSTEM CUSTOMIZATION
// ===================================================================

/**
 * Default system settings
 */
const defaultColors = {
  menuBackground: "#8B4513",
  menuActive: "#A0522D",
  menuHover: "#9B5A3C",
  cardBackground: "#FFFFFF",
  cardBorder: "#EEEEEE",
  primaryColor: "#8B4513",
  titleColor: "#333333",
  textColor: "#666666",
  buttonPrimary: "#8B4513",
  buttonSuccess: "#28a745",
  buttonDanger: "#dc3545",
  buttonWarning: "#ffc107",
};

const defaultSystemName = "Misayan Admin";

/**
 * Loads saved settings from database
 */
async function loadProfileSettings() {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/profile-settings`
    );
    if (response.ok) {
      const data = await response.json();

      // Load and apply colors
      let colors = defaultColors;
      if (data.customColors) {
        colors = JSON.parse(data.customColors);
      }
      applyColors(colors);
      loadColorsToInputs(colors);

      // Load system name
      const systemNameInput = document.getElementById("systemName");
      if (data.systemName) {
        updateSystemNameDisplay(data.systemName);
        if (systemNameInput) {
          systemNameInput.value = data.systemName;
        }
      } else {
        // If no name saved, use default
        if (systemNameInput) {
          systemNameInput.value = "Misayan Admin";
        }
      }

      // Load login logo
      if (data.loginLogo) {
        const loginLogoPreview = document.getElementById("loginLogoPreview");
        if (loginLogoPreview) {
          loginLogoPreview.src = data.loginLogo;
          loginLogoPreview.style.display = "block";
        }
        const removeBtn = document.getElementById("removeLoginLogoBtn");
        if (removeBtn) {
          removeBtn.style.display = "inline-block";
        }
      }

      // Load system logo
      if (data.systemLogo) {
        const systemLogoPreview = document.getElementById("systemLogoPreview");
        if (systemLogoPreview) {
          systemLogoPreview.src = data.systemLogo;
          systemLogoPreview.style.display = "block";
        }
        const removeBtn = document.getElementById("removeSystemLogoBtn");
        if (removeBtn) {
          removeBtn.style.display = "inline-block";
        }
        updateSystemLogoDisplay(data.systemLogo);
      }

      // Load favicon
      if (data.favicon) {
        const faviconPreview = document.getElementById("faviconPreview");
        if (faviconPreview) {
          faviconPreview.src = data.favicon;
          faviconPreview.style.display = "block";
        }
        const removeBtn = document.getElementById("removeFaviconBtn");
        if (removeBtn) {
          removeBtn.style.display = "inline-block";
        }
        updateFaviconDisplay(data.favicon);
      }

      // Load login colors
      loadLoginColorsToInputs(data.loginBodyColor, data.loginCardColor);
    } else {
      loadColorsToInputs(defaultColors);
      loadLoginColorsToInputs(null, null);
    }
  } catch (e) {
    console.error("Erro ao carregar configurações:", e);
    loadColorsToInputs(defaultColors);
  }
}

/**
 * Carrega as cores nos inputs da página de edição
 */
function loadColorsToInputs(colors) {
  console.log("loadColorsToInputs chamada com:", colors);

  Object.keys(colors).forEach((key) => {
    const colorInput = document.getElementById(key);
    const textInput = document.getElementById(key + "Text");

    console.log(`Processando ${key}:`, {
      colorInput: !!colorInput,
      textInput: !!textInput,
      value: colors[key],
    });

    if (colorInput) {
      colorInput.value = colors[key];
    }
    if (textInput) {
      textInput.value = colors[key].toUpperCase();
    }
  });

  console.log("loadColorsToInputs concluída");
}

/**
 * Sincroniza o input de cor com o input de texto
 */
function syncColorInputs() {
  const colorInputs = document.querySelectorAll('input[type="color"]');

  colorInputs.forEach((input) => {
    const textInput = document.getElementById(input.id + "Text");

    if (textInput) {
      // Quando o color picker muda, atualiza o texto
      input.addEventListener("input", () => {
        textInput.value = input.value.toUpperCase();
      });

      // Quando o texto muda, atualiza o color picker
      textInput.addEventListener("input", () => {
        const value = textInput.value.trim();
        if (/^#[0-9A-F]{6}$/i.test(value)) {
          input.value = value;
        }
      });
    }
  });
}

/**
 * Preview de uma cor específica
 */
function previewColor(colorKey) {
  const colorInput = document.getElementById(colorKey);
  const color = colorInput.value;

  const previewMap = {
    menuBackground: () => {
      document.querySelector(".preview-menu").style.background = color;
    },
    menuActive: () => {
      document.querySelector(".preview-menu-item.active").style.background =
        color;
    },
    menuHover: () => {
      document
        .querySelectorAll(".preview-menu-item:not(.active)")
        .forEach((item) => {
          item.style.background = color;
        });
    },
    cardBackground: () => {
      document.querySelector(".preview-card").style.background = color;
    },
    cardBorder: () => {
      document.querySelector(".preview-card").style.borderColor = color;
    },
    primaryColor: () => {
      document.querySelector(".preview-card h3").style.color = color;
    },
    titleColor: () => {
      document.querySelector(".preview-card h3").style.color = color;
    },
    textColor: () => {
      document.querySelector(".preview-card p").style.color = color;
    },
    buttonPrimary: () => {
      document.querySelector(".preview-btn-primary").style.background = color;
    },
    buttonSuccess: () => {
      document.querySelector(".preview-btn-success").style.background = color;
    },
    buttonDanger: () => {
      document.querySelector(".preview-btn-danger").style.background = color;
    },
    buttonWarning: () => {
      document.querySelector(".preview-btn-warning").style.background = color;
    },
  };

  if (previewMap[colorKey]) {
    previewMap[colorKey]();
  }

  showNotification(`Preview aplicado para ${colorKey}`, "info");
}

/**
 * Aplica as cores no sistema inteiro
 */
function applyColors(colors) {
  const root = document.documentElement;

  // Aplicar cores via CSS custom properties
  root.style.setProperty("--primary-color", colors.primaryColor);
  root.style.setProperty("--menu-background", colors.menuBackground);
  root.style.setProperty("--menu-active", colors.menuActive);
  root.style.setProperty("--menu-hover", colors.menuHover);
  root.style.setProperty("--card-background", colors.cardBackground);
  root.style.setProperty("--card-border", colors.cardBorder);
  root.style.setProperty("--title-color", colors.titleColor);
  root.style.setProperty("--text-color", colors.textColor);
  root.style.setProperty("--button-primary", colors.buttonPrimary);
  root.style.setProperty("--button-success", colors.buttonSuccess);
  root.style.setProperty("--button-danger", colors.buttonDanger);
  root.style.setProperty("--button-warning", colors.buttonWarning);

  // Aplicar cores diretamente em elementos específicos
  const sidebar = document.querySelector(".sidebar");
  if (sidebar) {
    sidebar.style.background = colors.menuBackground;
  }

  const navItems = document.querySelectorAll(".nav-item");
  navItems.forEach((item) => {
    item.addEventListener("mouseenter", function () {
      if (!this.classList.contains("active")) {
        this.style.background = colors.menuHover;
      }
    });
    item.addEventListener("mouseleave", function () {
      if (!this.classList.contains("active")) {
        this.style.background = "";
      }
    });
  });

  const activeNavItem = document.querySelector(".nav-item.active");
  if (activeNavItem) {
    activeNavItem.style.background = colors.menuActive;
  }

  // Aplicar cores nos cards
  const cards = document.querySelectorAll(
    ".stat-card, .chart-container, .store-card, .metric-card"
  );
  cards.forEach((card) => {
    card.style.background = colors.cardBackground;
    card.style.borderColor = colors.cardBorder;
  });

  // Aplicar cores nos títulos
  const titles = document.querySelectorAll("h1, h2, h3");
  titles.forEach((title) => {
    if (!title.closest(".sidebar")) {
      title.style.color = colors.titleColor;
    }
  });

  // Aplicar cores nos botões
  const primaryButtons = document.querySelectorAll(".btn-primary");
  primaryButtons.forEach((btn) => {
    btn.style.background = colors.buttonPrimary;
  });

  const successButtons = document.querySelectorAll(".btn-success");
  successButtons.forEach((btn) => {
    btn.style.background = colors.buttonSuccess;
  });

  const dangerButtons = document.querySelectorAll(".btn-danger");
  dangerButtons.forEach((btn) => {
    btn.style.background = colors.buttonDanger;
  });

  const warningButtons = document.querySelectorAll(".btn-warning");
  warningButtons.forEach((btn) => {
    btn.style.background = colors.buttonWarning;
  });
}

/**
 * Salva as cores do menu
 */
async function saveMenuColors() {
  const colors = getCurrentColors();
  colors.menuBackground = document.getElementById("menuBackground").value;
  colors.menuActive = document.getElementById("menuActive").value;
  colors.menuHover = document.getElementById("menuHover").value;

  await saveColorsToServer(colors);
}

/**
 * Salva as cores dos cards
 */
async function saveCardColors() {
  const colors = getCurrentColors();
  colors.cardBackground = document.getElementById("cardBackground").value;
  colors.cardBorder = document.getElementById("cardBorder").value;

  await saveColorsToServer(colors);
}

/**
 * Salva as cores dos textos
 */
async function saveTextColors() {
  const colors = getCurrentColors();
  colors.primaryColor = document.getElementById("primaryColor").value;
  colors.titleColor = document.getElementById("titleColor").value;
  colors.textColor = document.getElementById("textColor").value;

  await saveColorsToServer(colors);
}

/**
 * Salva as cores dos botões
 */
async function saveButtonColors() {
  const colors = getCurrentColors();
  colors.buttonPrimary = document.getElementById("buttonPrimary").value;
  colors.buttonSuccess = document.getElementById("buttonSuccess").value;
  colors.buttonDanger = document.getElementById("buttonDanger").value;
  colors.buttonWarning = document.getElementById("buttonWarning").value;

  await saveColorsToServer(colors);
}

/**
 * Obtém todas as cores atuais dos inputs
 */
function getCurrentColors() {
  return {
    menuBackground: document.getElementById("menuBackground").value,
    menuActive: document.getElementById("menuActive").value,
    menuHover: document.getElementById("menuHover").value,
    cardBackground: document.getElementById("cardBackground").value,
    cardBorder: document.getElementById("cardBorder").value,
    primaryColor: document.getElementById("primaryColor").value,
    titleColor: document.getElementById("titleColor").value,
    textColor: document.getElementById("textColor").value,
    buttonPrimary: document.getElementById("buttonPrimary").value,
    buttonSuccess: document.getElementById("buttonSuccess").value,
    buttonDanger: document.getElementById("buttonDanger").value,
    buttonWarning: document.getElementById("buttonWarning").value,
  };
}

/**
 * Salva as cores no servidor
 */
async function saveColorsToServer(colors) {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/profile-settings`,
      {
        method: "POST",
        body: JSON.stringify({ customColors: JSON.stringify(colors) }),
      }
    );

    if (response.ok) {
      applyColors(colors);
      updatePreviewRealtime();
      showNotification("Cores salvas com sucesso!", "success");
    } else {
      showNotification("Erro ao salvar cores", "error");
    }
  } catch (error) {
    console.error("Erro ao salvar cores:", error);
    showNotification("Erro ao salvar cores", "error");
  }
}

/**
 * Salva todas as cores (função mantida para compatibilidade)
 */
async function saveColors() {
  const colors = getCurrentColors();
  await saveColorsToServer(colors);
}

/**
 * Restaura as cores padrão
 */
async function resetColors() {
  if (
    confirm(
      "Tem certeza que deseja restaurar as cores padrão? Esta ação não pode ser desfeita."
    )
  ) {
    try {
      const response = await authenticatedFetch(
        `${API_BASE}/api/admin/profile-settings`,
        {
          method: "POST",
          body: JSON.stringify({ customColors: null }),
        }
      );

      if (response.ok) {
        applyColors(defaultColors);
        loadColorsToInputs(defaultColors);

        // Atualizar preview
        Object.keys(defaultColors).forEach((key) => {
          previewColor(key);
        });

        showNotification("Cores restauradas para o padrão!", "success");
      } else {
        showNotification("Erro ao restaurar cores", "error");
      }
    } catch (error) {
      console.error("Erro ao restaurar cores:", error);
      showNotification("Erro ao restaurar cores", "error");
    }
  }
}

/**
 * Atualiza o nome do sistema
 */
async function updateSystemName() {
  const systemName = document.getElementById("systemName").value.trim();

  if (!systemName) {
    showNotification("Por favor, insira um nome para o sistema.", "error");
    return;
  }

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/profile-settings`,
      {
        method: "POST",
        body: JSON.stringify({ systemName: systemName }),
      }
    );

    if (response.ok) {
      updateSystemNameDisplay(systemName);
      showNotification("Nome do sistema atualizado com sucesso!", "success");
    } else {
      showNotification("Erro ao atualizar nome do sistema", "error");
    }
  } catch (error) {
    console.error("Erro ao atualizar nome do sistema:", error);
    showNotification("Erro ao atualizar nome do sistema", "error");
  }
}

/**
 * Atualiza a exibição do nome do sistema em todos os lugares
 */
function updateSystemNameDisplay(name) {
  // Atualizar no logo da sidebar
  const logoElement = document.querySelector(".sidebar .logo h2");
  if (logoElement) {
    logoElement.textContent = name;
  }

  // Atualizar no preview
  const previewName = document.getElementById("previewSystemName");
  if (previewName) {
    previewName.textContent = name;
  }

  // Atualizar no título da página
  document.title = name;
}

/**
 * Inicialização quando a página de perfil é carregada
 */
function initProfilePage() {
  console.log("initProfilePage chamada");

  // Pequeno delay para garantir que todos os elementos estejam renderizados
  setTimeout(() => {
    // Sempre recarregar as configurações do servidor quando a página é exibida
    loadProfileSettings();
    syncColorInputs();
    syncLoginColorInputs();
  }, 100);
}

// Inicializar quando o DOM estiver pronto
document.addEventListener("DOMContentLoaded", () => {
  // Carregar configurações salvas ao iniciar
  loadProfileSettings();

  // Verificar se estamos na página de perfil
  const profilePage = document.getElementById("edit-profile-page");
  if (profilePage) {
    // Observar quando a página de perfil é exibida
    const observer = new MutationObserver((mutations) => {
      mutations.forEach((mutation) => {
        if (mutation.attributeName === "style") {
          const display = window.getComputedStyle(profilePage).display;
          if (display !== "none") {
            initProfilePage();
          }
        }
      });
    });

    observer.observe(profilePage, { attributes: true });
  }
});

/**
 * Abre o modal de preview lateral
 */
function openPreviewModal(type) {
  const modal = document.getElementById("colorPreviewModal");
  const modalBody = document.getElementById("previewModalBody");
  const modalTitle = document.getElementById("previewModalTitle");

  // Definir título baseado no tipo
  const titles = {
    menu: "Preview - Cores do Menu",
    cards: "Preview - Cores dos Cards",
    texts: "Preview - Cores dos Textos",
    buttons: "Preview - Cores dos Botões",
  };

  modalTitle.textContent = titles[type] || "Preview em Tempo Real";

  // Gerar conteúdo do preview baseado no tipo
  modalBody.innerHTML = generatePreviewContent(type);

  // Aplicar cores atuais no preview
  updatePreviewRealtime();

  // Mostrar modal
  modal.style.display = "flex";

  // Adicionar classe para animação
  setTimeout(() => {
    modal.querySelector(".preview-modal-sidebar-content").style.transform =
      "translateX(0)";
  }, 10);
}

/**
 * Fecha o modal de preview
 */
function closePreviewModal() {
  const modal = document.getElementById("colorPreviewModal");
  const sidebarContent = modal.querySelector(".preview-modal-sidebar-content");

  sidebarContent.style.transform = "translateX(-100%)";

  setTimeout(() => {
    modal.style.display = "none";
  }, 300);
}

/**
 * Gera o conteúdo do preview baseado no tipo
 */
function generatePreviewContent(type) {
  const colors = getCurrentColors();

  switch (type) {
    case "menu":
      return `
                <div class="preview-menu" style="background: ${colors.menuBackground}; padding: 20px; border-radius: 8px;">
                    <h3 style="color: white; margin-bottom: 15px;">Menu Lateral</h3>
                    <div class="preview-menu-item" style="padding: 12px; margin-bottom: 8px; border-radius: 6px; color: white; cursor: pointer;">
                        📊 Dashboard
                    </div>
                    <div class="preview-menu-item active" style="padding: 12px; margin-bottom: 8px; border-radius: 6px; background: ${colors.menuActive}; color: white; cursor: pointer;">
                        👤 Editar Perfil
                    </div>
                    <div class="preview-menu-item" style="padding: 12px; margin-bottom: 8px; border-radius: 6px; color: white; cursor: pointer;">
                        💬 Conversas
                    </div>
                    <div class="preview-menu-item" style="padding: 12px; border-radius: 6px; color: white; cursor: pointer;">
                        👥 Usuários
                    </div>
                </div>
            `;
    case "cards":
      return `
                <div class="preview-card" style="background: ${colors.cardBackground}; border: 2px solid ${colors.cardBorder}; padding: 20px; border-radius: 8px; margin-bottom: 15px;">
                    <h3 style="color: ${colors.titleColor}; margin-bottom: 10px;">Card de Exemplo</h3>
                    <p style="color: ${colors.textColor}; line-height: 1.6;">
                        Este é um exemplo de como os cards ficarão com as cores selecionadas. 
                        Os cards são usados para exibir informações importantes no sistema.
                    </p>
                </div>
                <div class="preview-card" style="background: ${colors.cardBackground}; border: 2px solid ${colors.cardBorder}; padding: 20px; border-radius: 8px;">
                    <h3 style="color: ${colors.titleColor}; margin-bottom: 10px;">Outro Card</h3>
                    <p style="color: ${colors.textColor}; line-height: 1.6;">
                        Múltiplos cards podem ser exibidos lado a lado ou empilhados.
                    </p>
                </div>
            `;
    case "texts":
      return `
                <div style="padding: 20px; background: white; border-radius: 8px;">
                    <h1 style="color: ${colors.titleColor}; margin-bottom: 15px;">Título Principal</h1>
                    <h2 style="color: ${colors.titleColor}; margin-bottom: 12px;">Subtítulo</h2>
                    <p style="color: ${colors.textColor}; line-height: 1.8; margin-bottom: 15px;">
                        Este é um parágrafo de exemplo mostrando como o texto normal será exibido no sistema.
                        A cor do texto é importante para garantir boa legibilidade.
                    </p>
                    <p style="color: ${colors.primaryColor}; font-weight: bold;">
                        Texto com cor primária em destaque
                    </p>
                </div>
            `;
    case "buttons":
      return `
                <div style="padding: 20px; background: white; border-radius: 8px;">
                    <h3 style="margin-bottom: 20px; color: #333;">Exemplos de Botões</h3>
                    <button class="preview-btn-primary" style="background: ${colors.buttonPrimary}; color: white; border: none; padding: 12px 24px; border-radius: 6px; margin: 5px; cursor: pointer; font-size: 14px;">
                        Botão Primário
                    </button>
                    <button class="preview-btn-success" style="background: ${colors.buttonSuccess}; color: white; border: none; padding: 12px 24px; border-radius: 6px; margin: 5px; cursor: pointer; font-size: 14px;">
                        Botão Sucesso
                    </button>
                    <button class="preview-btn-danger" style="background: ${colors.buttonDanger}; color: white; border: none; padding: 12px 24px; border-radius: 6px; margin: 5px; cursor: pointer; font-size: 14px;">
                        Botão Perigo
                    </button>
                    <button class="preview-btn-warning" style="background: ${colors.buttonWarning}; color: white; border: none; padding: 12px 24px; border-radius: 6px; margin: 5px; cursor: pointer; font-size: 14px;">
                        Botão Aviso
                    </button>
                </div>
            `;
    default:
      return "<p>Preview não disponível</p>";
  }
}

/**
 * Atualiza o preview em tempo real
 */
function updatePreviewRealtime() {
  const modal = document.getElementById("colorPreviewModal");
  if (modal && modal.style.display === "flex") {
    const modalBody = document.getElementById("previewModalBody");
    const currentContent = modalBody.innerHTML;

    // Detectar tipo de preview atual
    let type = "menu";
    if (currentContent.includes("preview-card")) type = "cards";
    else if (currentContent.includes("Título Principal")) type = "texts";
    else if (currentContent.includes("Exemplos de Botões")) type = "buttons";

    // Regenerar conteúdo com cores atualizadas
    modalBody.innerHTML = generatePreviewContent(type);
  }
}

/**
 * Funções de upload e gerenciamento de logos
 */
let currentCropper = null;
let currentLogoType = null;

async function handleLoginLogoUpload(event) {
  const file = event.target.files[0];
  if (!file) return;

  currentLogoType = "loginLogo";
  openCropModal(file, 140, 140);
}

async function handleSystemLogoUpload(event) {
  const file = event.target.files[0];
  if (!file) return;

  currentLogoType = "systemLogo";
  openCropModal(file, 210, 70);
}

async function handleFaviconUpload(event) {
  const file = event.target.files[0];
  if (!file) return;

  currentLogoType = "favicon";
  openCropModal(file, 64, 64);
}

function openCropModal(file, maxWidth, maxHeight) {
  const modal = document.getElementById("cropModal");
  const image = document.getElementById("cropImage");
  const reader = new FileReader();

  reader.onload = function (e) {
    image.src = e.target.result;
    modal.style.display = "flex";

    // Destruir cropper anterior se existir
    if (currentCropper) {
      currentCropper.destroy();
    }

    // Inicializar Cropper.js
    currentCropper = new Cropper(image, {
      aspectRatio: maxWidth / maxHeight,
      viewMode: 1,
      autoCropArea: 1,
      responsive: true,
      background: false,
      modal: true,
      guides: true,
      center: true,
      highlight: true,
      cropBoxResizable: true,
      cropBoxMovable: true,
      toggleDragModeOnDblclick: false,
    });
  };

  reader.readAsDataURL(file);
}

function closeCropModal() {
  const modal = document.getElementById("cropModal");
  modal.style.display = "none";

  if (currentCropper) {
    currentCropper.destroy();
    currentCropper = null;
  }

  // Limpar input de arquivo
  document.getElementById("loginLogoInput").value = "";
  document.getElementById("systemLogoInput").value = "";
  const faviconInput = document.getElementById("faviconInput");
  if (faviconInput) {
    faviconInput.value = "";
  }
}

async function confirmCrop() {
  if (!currentCropper) return;

  // Obter dimensões máximas baseado no tipo de logo
  let maxWidth, maxHeight;
  if (currentLogoType === "loginLogo") {
    maxWidth = 140;
    maxHeight = 140;
  } else if (currentLogoType === "systemLogo") {
    maxWidth = 210;
    maxHeight = 70;
  } else if (currentLogoType === "favicon") {
    maxWidth = 64;
    maxHeight = 64;
  }

  // Obter imagem recortada
  const canvas = currentCropper.getCroppedCanvas({
    maxWidth: maxWidth,
    maxHeight: maxHeight,
    imageSmoothingEnabled: true,
    imageSmoothingQuality: "high",
  });

  const croppedImage = canvas.toDataURL("image/png");

  // Mostrar preview
  let previewId, removeBtnId;
  if (currentLogoType === "loginLogo") {
    previewId = "loginLogoPreview";
    removeBtnId = "removeLoginLogoBtn";
  } else if (currentLogoType === "systemLogo") {
    previewId = "systemLogoPreview";
    removeBtnId = "removeSystemLogoBtn";
  } else if (currentLogoType === "favicon") {
    previewId = "faviconPreview";
    removeBtnId = "removeFaviconBtn";
  }

  const preview = document.getElementById(previewId);
  preview.src = croppedImage;
  preview.style.display = "block";
  document.getElementById(removeBtnId).style.display = "inline-block";

  // Salvar no servidor
  await saveLogoToServer(currentLogoType, croppedImage);

  // Atualizar favicon no navegador se for o caso
  if (currentLogoType === "favicon") {
    updateFaviconDisplay(croppedImage);
  }

  // Fechar modal
  closeCropModal();
}

async function saveLogoToServer(logoType, logoData) {
  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/profile-settings`,
      {
        method: "POST",
        body: JSON.stringify({ [logoType]: logoData }),
      }
    );

    if (response.ok) {
      showNotification("Logo salva com sucesso!", "success");
      if (logoType === "systemLogo") {
        updateSystemLogoDisplay(logoData);
      }
    } else {
      showNotification("Erro ao salvar logo", "error");
    }
  } catch (error) {
    console.error("Erro ao salvar logo:", error);
    showNotification("Erro ao salvar logo", "error");
  }
}

async function removeLoginLogo() {
  if (confirm("Tem certeza que deseja remover a logo de login?")) {
    try {
      const response = await authenticatedFetch(
        `${API_BASE}/api/admin/profile-settings`,
        {
          method: "POST",
          body: JSON.stringify({ loginLogo: null }),
        }
      );

      if (response.ok) {
        document.getElementById("loginLogoPreview").style.display = "none";
        document.getElementById("removeLoginLogoBtn").style.display = "none";
        showNotification("Logo removida com sucesso!", "success");
      }
    } catch (error) {
      console.error("Erro ao remover logo:", error);
      showNotification("Erro ao remover logo", "error");
    }
  }
}

async function removeSystemLogo() {
  if (confirm("Tem certeza que deseja remover a logo do sistema?")) {
    try {
      const response = await authenticatedFetch(
        `${API_BASE}/api/admin/profile-settings`,
        {
          method: "POST",
          body: JSON.stringify({ systemLogo: null }),
        }
      );

      if (response.ok) {
        document.getElementById("systemLogoPreview").style.display = "none";
        document.getElementById("removeSystemLogoBtn").style.display = "none";
        updateSystemLogoDisplay(null);
        showNotification("Logo removida com sucesso!", "success");
      }
    } catch (error) {
      console.error("Erro ao remover logo:", error);
      showNotification("Erro ao remover logo", "error");
    }
  }
}

async function removeFavicon() {
  if (confirm("Tem certeza que deseja remover o favicon?")) {
    try {
      const response = await authenticatedFetch(
        `${API_BASE}/api/admin/profile-settings`,
        {
          method: "POST",
          body: JSON.stringify({ favicon: null }),
        }
      );

      if (response.ok) {
        document.getElementById("faviconPreview").style.display = "none";
        document.getElementById("removeFaviconBtn").style.display = "none";
        updateFaviconDisplay(null);
        showNotification("Favicon removido com sucesso!", "success");
      }
    } catch (error) {
      console.error("Erro ao remover favicon:", error);
      showNotification("Erro ao remover favicon", "error");
    }
  }
}

function updateFaviconDisplay(faviconData) {
  // Atualizar favicon no head da página
  let link = document.querySelector("link[rel~='icon']");
  if (!link) {
    link = document.createElement("link");
    link.rel = "icon";
    document.head.appendChild(link);
  }

  if (faviconData) {
    link.href = faviconData;
  } else {
    link.href = "/favicon.ico"; // Favicon padrão
  }
}

function updateSystemLogoDisplay(logoData) {
  const logoElement = document.querySelector(".sidebar .logo");
  if (logoElement) {
    if (logoData) {
      logoElement.innerHTML = `<img src="${logoData}" alt="Logo" style="max-width: 210px; max-height: 70px;" />`;
    } else {
      const systemName =
        document.getElementById("systemName")?.value || "Misayan Admin";
      logoElement.innerHTML = `<h2>${systemName}</h2>`;
    }
  }
}

/**
 * Abre o modal de preview da tela de login
 */
function openLoginPreviewModal() {
  const modal = document.getElementById("loginPreviewModal");
  const container = document.getElementById("loginPreviewContainer");

  // Gerar preview da tela de login
  updateLoginPreview("mobile");

  modal.style.display = "flex";
}

/**
 * Fecha o modal de preview da tela de login
 */
function closeLoginPreviewModal() {
  const modal = document.getElementById("loginPreviewModal");
  modal.style.display = "none";
}

/**
 * Alterna entre dispositivos no preview de login
 */
function switchLoginPreviewDevice(device) {
  const container = document.getElementById("loginPreviewContainer");
  const buttons = document.querySelectorAll(".device-btn");

  // Atualizar botões ativos
  buttons.forEach((btn) => btn.classList.remove("active"));
  document.querySelector(`[data-device="${device}"]`).classList.add("active");

  // Atualizar classe do container
  container.className = `login-preview-frame ${device}`;

  // Atualizar preview
  updateLoginPreview(device);
}

/**
 * Atualiza o preview da tela de login
 */
function updateLoginPreview(device) {
  const container = document.getElementById("loginPreviewContainer");
  const systemName =
    document.getElementById("systemName")?.value || "Misayan Admin";
  const loginLogo = document.getElementById("loginLogoPreview")?.src || "";
  const colors = getCurrentColors();

  const logoHtml = loginLogo
    ? `<img src="${loginLogo}" alt="Logo" style="max-width: 140px; max-height: 140px; margin-bottom: 20px;" />`
    : "";

  container.innerHTML = `
        <div class="login-preview-box" style="background: white; padding: 40px; border-radius: 12px; box-shadow: 0 4px 20px rgba(0,0,0,0.1); text-align: center; max-width: 400px; margin: auto;">
            ${logoHtml}
            <h1 style="color: ${
              colors.titleColor
            }; margin-bottom: 10px; font-size: ${
    device === "mobile" ? "24px" : "32px"
  };">${systemName}</h1>
            <p style="color: ${
              colors.textColor
            }; margin-bottom: 30px; font-size: ${
    device === "mobile" ? "14px" : "16px"
  };">Sistema de Chatbot WhatsApp</p>
            
            <div style="margin-bottom: 15px;">
                <input type="text" placeholder="Usuário" style="width: 100%; padding: 12px; border: 2px solid ${
                  colors.cardBorder
                }; border-radius: 6px; font-size: 14px;" />
            </div>
            
            <div style="margin-bottom: 20px;">
                <input type="password" placeholder="Senha" style="width: 100%; padding: 12px; border: 2px solid ${
                  colors.cardBorder
                }; border-radius: 6px; font-size: 14px;" />
            </div>
            
            <button style="width: 100%; padding: 14px; background: ${
              colors.buttonPrimary
            }; color: white; border: none; border-radius: 6px; font-size: 16px; font-weight: bold; cursor: pointer;">
                Entrar
            </button>
        </div>
    `;
}

// Exportar funções para uso global
window.previewColor = previewColor;
window.saveColors = saveColors;
window.saveMenuColors = saveMenuColors;
window.saveCardColors = saveCardColors;
window.saveTextColors = saveTextColors;
window.saveButtonColors = saveButtonColors;
window.resetColors = resetColors;
window.updateSystemName = updateSystemName;
window.openPreviewModal = openPreviewModal;
window.closePreviewModal = closePreviewModal;
window.updatePreviewRealtime = updatePreviewRealtime;
window.handleLoginLogoUpload = handleLoginLogoUpload;
window.handleSystemLogoUpload = handleSystemLogoUpload;
window.handleFaviconUpload = handleFaviconUpload;
window.removeLoginLogo = removeLoginLogo;
window.removeSystemLogo = removeSystemLogo;
window.removeFavicon = removeFavicon;
window.openLoginPreviewModal = openLoginPreviewModal;
window.closeLoginPreviewModal = closeLoginPreviewModal;
window.switchLoginPreviewDevice = switchLoginPreviewDevice;
window.closeCropModal = closeCropModal;
window.confirmCrop = confirmCrop;

// ===================================================================
//  FUNÇÕES PARA CORES DA PÁGINA DE LOGIN
// ===================================================================

/**
 * Sincroniza os inputs de cor de login (color e text)
 */
function syncLoginColorInputs() {
  const bodyColor = document.getElementById("loginBodyColor");
  const bodyColorText = document.getElementById("loginBodyColorText");
  const cardColor = document.getElementById("loginCardColor");
  const cardColorText = document.getElementById("loginCardColorText");

  if (bodyColor && bodyColorText) {
    bodyColor.addEventListener("input", (e) => {
      bodyColorText.value = e.target.value.toUpperCase();
    });

    bodyColorText.addEventListener("input", (e) => {
      const value = e.target.value;
      if (/^#[0-9A-F]{6}$/i.test(value)) {
        bodyColor.value = value;
      }
    });
  }

  if (cardColor && cardColorText) {
    cardColor.addEventListener("input", (e) => {
      cardColorText.value = e.target.value.toUpperCase();
    });

    cardColorText.addEventListener("input", (e) => {
      const value = e.target.value;
      if (/^#[0-9A-F]{6}$/i.test(value)) {
        cardColor.value = value;
      }
    });
  }
}

/**
 * Salva as cores da página de login
 */
async function saveLoginColors() {
  const bodyColor = document.getElementById("loginBodyColor").value;
  const cardColor = document.getElementById("loginCardColor").value;

  try {
    const response = await authenticatedFetch(
      `${API_BASE}/api/admin/profile-settings`,
      {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          loginBodyColor: bodyColor,
          loginCardColor: cardColor,
        }),
      }
    );

    if (response.ok) {
      showNotification("Cores de login salvas com sucesso!", "success");
    } else {
      showNotification("Erro ao salvar cores de login", "error");
    }
  } catch (error) {
    console.error("Erro ao salvar cores de login:", error);
    showNotification("Erro ao salvar cores de login", "error");
  }
}

/**
 * Carrega as cores de login salvas nos inputs
 */
function loadLoginColorsToInputs(loginBodyColor, loginCardColor) {
  const bodyColor = document.getElementById("loginBodyColor");
  const bodyColorText = document.getElementById("loginBodyColorText");
  const cardColor = document.getElementById("loginCardColor");
  const cardColorText = document.getElementById("loginCardColorText");

  if (bodyColor && bodyColorText) {
    const bodyValue = loginBodyColor || "#A0522D";
    bodyColor.value = bodyValue;
    bodyColorText.value = bodyValue.toUpperCase();
  }

  if (cardColor && cardColorText) {
    const cardValue = loginCardColor || "#FFFFFF";
    cardColor.value = cardValue;
    cardColorText.value = cardValue.toUpperCase();
  }
}
