// Payments Admin JavaScript

let paymentMethods = [];
let paymentLinks = [];
let currentPaymentUrl = '';

// Initialization
document.addEventListener('DOMContentLoaded', function() {
    loadPaymentMethods();
    loadPaymentLinks();
    
    // Event listeners
    document.getElementById('paymentForm').addEventListener('submit', handlePaymentSubmit);
    document.getElementById('configForm').addEventListener('submit', handleConfigSubmit);
    
    // Real-time search
    document.getElementById('linkSearch').addEventListener('input', debounce(searchLinks, 500));
});

// ===== TAB NAVIGATION =====

function showTab(tabName) {
    // Hide all tabs
    document.querySelectorAll('.tab-content').forEach(tab => {
        tab.classList.remove('active');
    });
    
    // Remove active class from all buttons
    document.querySelectorAll('.tab').forEach(btn => {
        btn.classList.remove('active');
    });
    
    // Show selected tab
    document.getElementById(tabName).classList.add('active');
    
    // Activate corresponding button
    event.target.classList.add('active');
    
    // Load tab-specific data
    switch(tabName) {
        case 'config':
            loadPaymentMethods();
            break;
        case 'create':
            populatePaymentMethods();
            break;
        case 'links':
            loadPaymentLinks();
            break;
    }
}

// ===== PAYMENT METHODS CONFIGURATION =====

async function loadPaymentMethods() {
    try {
        const token = localStorage.getItem('authToken');
        const response = await fetch('/api/payments/methods', {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            paymentMethods = await response.json();
            renderPaymentMethods();
        }
    } catch (error) {
        console.error('Error loading payment methods:', error);
        showNotification('Error loading payment methods', 'error');
    }
}

function renderPaymentMethods() {
    const container = document.getElementById('paymentMethods');
    container.innerHTML = '';
    
    const methods = [
        { name: 'paypal', displayName: 'PayPal', logo: 'PP' },
        { name: 'pagseguro', displayName: 'PagSeguro', logo: 'PS' }
    ];
    
    methods.forEach(method => {
        const configuredMethod = paymentMethods.find(pm => pm.method_name === method.name);
        
        const methodDiv = document.createElement('div');
        methodDiv.className = 'payment-method-card';
        
        let status = 'not-configured';
        let statusText = 'Not Configured';
        
        if (configuredMethod) {
            status = configuredMethod.active ? 'active' : 'inactive';
            statusText = configuredMethod.active ? 'Active' : 'Inactive';
        }
        
        methodDiv.innerHTML = `
            <div class="method-header">
                <div class="method-name">
                    <div class="method-logo ${method.name}-logo">${method.logo}</div>
                    ${method.displayName}
                </div>
                <span class="status-badge status-${status}">${statusText}</span>
            </div>
            
            <p>
                ${method.name === 'paypal' ? 
                    'Aceite pagamentos via PayPal com segurança e facilidade.' :
                    'Processe pagamentos com PagSeguro, líder no Brasil.'
                }
            </p>
            
            <div style="margin-top: 15px;">
                <button class="btn btn-primary" onclick="configureMethod('${method.name}')">
                    ⚙️ ${configuredMethod ? 'Reconfigurar' : 'Configurar'}
                </button>
                
                ${configuredMethod ? `
                    <button class="btn ${configuredMethod.active ? 'btn-warning' : 'btn-success'}" 
                            onclick="toggleMethod(${configuredMethod.id}, ${!configuredMethod.active})">
                        ${configuredMethod.active ? '⏸️ Desativar' : '▶️ Ativar'}
                    </button>
                ` : ''}
            </div>
            
            ${configuredMethod ? `
                <div style="margin-top: 10px; font-size: 12px; color: #666;">
                    <strong>Modo:</strong> ${configuredMethod.sandbox_mode ? 'Sandbox (Teste)' : 'Produção'}<br>
                    <strong>Configurado em:</strong> ${new Date(configuredMethod.created_at).toLocaleString('pt-BR')}
                </div>
            ` : ''}
        `;
        
        container.appendChild(methodDiv);
    });
}

function configureMethod(methodName) {
    const configuredMethod = paymentMethods.find(pm => pm.method_name === methodName);
    
    document.getElementById('configMethod').value = methodName;
    document.getElementById('configModalTitle').textContent = 
        `Configurar ${methodName === 'paypal' ? 'PayPal' : 'PagSeguro'}`;
    
    // Ajustar campos baseado no método
    const apiKeyLabel = document.querySelector('label[for="configApiKey"]');
    const secretLabel = document.querySelector('label[for="configApiSecret"]');
    const secretGroup = document.getElementById('secretGroup');
    const apiKeyInput = document.getElementById('configApiKey');
    const apiSecretInput = document.getElementById('configApiSecret');
    
    if (methodName === 'paypal') {
        apiKeyLabel.textContent = 'Client ID';
        secretLabel.textContent = 'Client Secret';
        secretGroup.style.display = 'block';
        apiSecretInput.required = true;
        
        // Preencher com valores existentes se houver
        if (configuredMethod) {
            apiKeyInput.value = configuredMethod.api_key || '';
            apiSecretInput.value = configuredMethod.api_secret || '';
            document.getElementById('configSandbox').checked = configuredMethod.sandbox_mode;
        } else {
            document.getElementById('configForm').reset();
            document.getElementById('configSandbox').checked = true;
        }
    } else {
        // PagSeguro
        apiKeyLabel.textContent = 'Email PagSeguro';
        secretLabel.textContent = 'Token PagSeguro';
        secretGroup.style.display = 'block';
        apiSecretInput.required = true;
        
        // Preencher com valores existentes se houver
        if (configuredMethod) {
            apiKeyInput.value = configuredMethod.api_key || '';
            apiSecretInput.value = configuredMethod.api_secret || '';
            document.getElementById('configSandbox').checked = configuredMethod.sandbox_mode;
        } else {
            document.getElementById('configForm').reset();
            document.getElementById('configSandbox').checked = true;
        }
    }
    
    document.getElementById('configModal').style.display = 'block';
}

async function handleConfigSubmit(e) {
    e.preventDefault();
    
    const methodName = document.getElementById('configMethod').value;
    const apiKey = document.getElementById('configApiKey').value;
    const apiSecret = document.getElementById('configApiSecret').value;
    const sandboxMode = document.getElementById('configSandbox').checked;
    
    try {
        const token = localStorage.getItem('authToken');
        const response = await fetch('/api/payments/methods', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({
                method_name: methodName,
                api_key: apiKey,
                api_secret: apiSecret,
                sandbox_mode: sandboxMode
            })
        });
        
        if (response.ok) {
            showNotification('Configuração salva com sucesso!', 'success');
            closeConfigModal();
            await loadPaymentMethods();
        } else {
            const error = await response.json();
            throw new Error(error.error || 'Erro ao salvar configuração');
        }
    } catch (error) {
        console.error('Erro ao salvar configuração:', error);
        showNotification(error.message, 'error');
    }
}

async function toggleMethod(methodId, active) {
    try {
        const token = localStorage.getItem('authToken');
        const response = await fetch(`/api/payments/methods/${methodId}/toggle`, {
            method: 'PATCH',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({ active })
        });
        
        if (response.ok) {
            showNotification(`Método ${active ? 'ativado' : 'desativado'} com sucesso!`, 'success');
            loadPaymentMethods();
            populatePaymentMethods();
        } else {
            const error = await response.json();
            throw new Error(error.error || 'Erro ao alterar status');
        }
    } catch (error) {
        console.error('Erro ao alterar status:', error);
        showNotification(error.message, 'error');
    }
}

function closeConfigModal() {
    document.getElementById('configModal').style.display = 'none';
}

// ===== CRIAÇÃO DE LINKS =====

function populatePaymentMethods() {
    const select = document.getElementById('paymentMethod');
    select.innerHTML = '<option value="">Selecione um método</option>';
    
    const activeMethods = paymentMethods.filter(method => method.active);
    
    activeMethods.forEach(method => {
        const option = document.createElement('option');
        option.value = method.method_name;
        option.textContent = method.method_name === 'paypal' ? 'PayPal' : 'PagSeguro';
        select.appendChild(option);
    });
    
    if (activeMethods.length === 0) {
        const option = document.createElement('option');
        option.value = '';
        option.textContent = 'Nenhum método ativo';
        option.disabled = true;
        select.appendChild(option);
    }
}

function updatePaymentPreview() {
    const method = document.getElementById('paymentMethod').value;
    const amount = document.getElementById('paymentAmount').value;
    const description = document.getElementById('paymentDescription').value;
    const customerPhone = document.getElementById('customerPhone').value;
    const customerName = document.getElementById('customerName').value;
    
    const preview = document.getElementById('paymentPreview');
    
    if (method && amount && customerPhone) {
        preview.style.display = 'block';
        
        document.getElementById('previewAmount').textContent = 
            `R$ ${parseFloat(amount).toLocaleString('pt-BR', { minimumFractionDigits: 2 })}`;
        
        document.getElementById('previewDescription').textContent = 
            description || 'Pagamento via WhatsApp';
        
        document.getElementById('previewCustomerName').textContent = 
            customerName || 'Cliente';
        
        document.getElementById('previewCustomerPhone').textContent = customerPhone;
    } else {
        preview.style.display = 'none';
    }
}

async function handlePaymentSubmit(e) {
    e.preventDefault();
    
    const method = document.getElementById('paymentMethod').value;
    const amount = document.getElementById('paymentAmount').value;
    const description = document.getElementById('paymentDescription').value;
    const customerPhone = document.getElementById('customerPhone').value;
    const customerName = document.getElementById('customerName').value;
    
    try {
        const token = localStorage.getItem('authToken');
        const response = await fetch('/api/payments/create-link', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({
                payment_method: method,
                amount: parseFloat(amount),
                description: description,
                customer_phone: customerPhone,
                customer_name: customerName
            })
        });
        
        if (response.ok) {
            const result = await response.json();
            showPaymentResult(result.payment_link);
            
            // Limpar formulário
            document.getElementById('paymentForm').reset();
            document.getElementById('paymentPreview').style.display = 'none';
            
        } else {
            const error = await response.json();
            throw new Error(error.error || 'Erro ao criar link de pagamento');
        }
    } catch (error) {
        console.error('Erro ao criar link:', error);
        showNotification(error.message, 'error');
    }
}

function showPaymentResult(paymentLink) {
    currentPaymentUrl = paymentLink.payment_url;
    
    document.getElementById('generatedUrl').textContent = currentPaymentUrl;
    document.getElementById('paymentResult').style.display = 'block';
    
    // Scroll para o resultado
    document.getElementById('paymentResult').scrollIntoView({ 
        behavior: 'smooth' 
    });
}

function copyPaymentUrl() {
    navigator.clipboard.writeText(currentPaymentUrl).then(() => {
        showNotification('Link copiado para a área de transferência!', 'success');
    }).catch(() => {
        showNotification('Erro ao copiar link', 'error');
    });
}

function sendPaymentLink() {
    const customerPhone = document.getElementById('customerPhone').value;
    const amount = document.getElementById('paymentAmount').value;
    const description = document.getElementById('paymentDescription').value || 'Pagamento';
    
    const message = `Olá! Aqui está o link para seu pagamento:\n\n` +
                   `💰 Valor: R$ ${parseFloat(amount).toLocaleString('pt-BR', { minimumFractionDigits: 2 })}\n` +
                   `📝 Descrição: ${description}\n\n` +
                   `🔗 Link: ${currentPaymentUrl}\n\n` +
                   `Clique no link para realizar o pagamento com segurança.`;
    
    const whatsappUrl = `https://wa.me/${customerPhone.replace(/\D/g, '')}?text=${encodeURIComponent(message)}`;
    window.open(whatsappUrl, '_blank');
}

function showQRCode() {
    const canvas = document.getElementById('qrCanvas');
    
    QRCode.toCanvas(canvas, currentPaymentUrl, {
        width: 200,
        margin: 2,
        color: {
            dark: '#000000',
            light: '#FFFFFF'
        }
    }, function (error) {
        if (error) {
            console.error('Erro ao gerar QR Code:', error);
            showNotification('Erro ao gerar QR Code', 'error');
        } else {
            document.getElementById('qrModal').style.display = 'block';
        }
    });
}

function closeQRModal() {
    document.getElementById('qrModal').style.display = 'none';
}

function downloadQRCode() {
    const canvas = document.getElementById('qrCanvas');
    const link = document.createElement('a');
    link.download = 'qrcode-pagamento.png';
    link.href = canvas.toDataURL();
    link.click();
}

// ===== GERENCIAMENTO DE LINKS =====

async function loadPaymentLinks(page = 1) {
    try {
        const token = localStorage.getItem('authToken');
        const search = document.getElementById('linkSearch')?.value || '';
        const status = document.getElementById('statusFilter')?.value || '';
        const method = document.getElementById('methodFilter')?.value || '';
        
        const params = new URLSearchParams({
            page: page,
            limit: 20,
            status: status,
            method: method
        });
        
        const response = await fetch(`/api/payments/links?${params}`, {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            const data = await response.json();
            paymentLinks = data.links;
            renderPaymentLinks(data);
        }
    } catch (error) {
        console.error('Erro ao carregar links:', error);
        showNotification('Erro ao carregar links de pagamento', 'error');
    }
}

function renderPaymentLinks(data) {
    const container = document.getElementById('linksTable');
    
    if (data.links.length === 0) {
        container.innerHTML = '<p>Nenhum link de pagamento encontrado.</p>';
        return;
    }
    
    let html = `
        <table class="links-table">
            <thead>
                <tr>
                    <th>Cliente</th>
                    <th>Valor</th>
                    <th>Método</th>
                    <th>Status</th>
                    <th>Criado</th>
                    <th>Ações</th>
                </tr>
            </thead>
            <tbody>
    `;
    
    data.links.forEach(link => {
        const statusClass = `status-${link.status}`;
        const statusText = getStatusText(link.status);
        
        html += `
            <tr>
                <td>
                    <strong>${link.customer_name || 'Cliente'}</strong><br>
                    <small>${link.customer_phone}</small>
                </td>
                <td class="amount-display">
                    R$ ${parseFloat(link.amount).toLocaleString('pt-BR', { minimumFractionDigits: 2 })}
                </td>
                <td>${link.payment_method === 'paypal' ? 'PayPal' : 'PagSeguro'}</td>
                <td>
                    <span class="status-badge ${statusClass}">${statusText}</span>
                </td>
                <td>${new Date(link.created_at).toLocaleString('pt-BR')}</td>
                <td class="link-actions">
                    <button class="btn btn-info" onclick="viewPaymentLink('${link.payment_url}')" title="Ver Link">👁️</button>
                    <button class="btn btn-primary" onclick="copyLinkUrl('${link.payment_url}')" title="Copiar">📋</button>
                    <button class="btn btn-warning" onclick="checkPaymentStatus(${link.id})" title="Verificar Status">🔄</button>
                </td>
            </tr>
        `;
    });
    
    html += '</tbody></table>';
    container.innerHTML = html;
    
    // Renderizar paginação
    renderPagination('linksPagination', data, loadPaymentLinks);
}

function searchLinks() {
    loadPaymentLinks(1);
}

function filterLinks() {
    loadPaymentLinks(1);
}

function viewPaymentLink(url) {
    window.open(url, '_blank');
}

function copyLinkUrl(url) {
    navigator.clipboard.writeText(url).then(() => {
        showNotification('Link copiado!', 'success');
    }).catch(() => {
        showNotification('Erro ao copiar link', 'error');
    });
}

async function checkPaymentStatus(linkId) {
    try {
        const token = localStorage.getItem('authToken');
        const response = await fetch(`/api/payments/links/${linkId}/status`, {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            const status = await response.json();
            showNotification(`Status: ${getStatusText(status.status)}`, 'info');
            
            // Recarregar lista se status mudou
            loadPaymentLinks();
        } else {
            throw new Error('Erro ao verificar status');
        }
    } catch (error) {
        console.error('Erro ao verificar status:', error);
        showNotification('Erro ao verificar status do pagamento', 'error');
    }
}

// ===== UTILITÁRIOS =====

function getStatusText(status) {
    const texts = {
        pending: 'Pendente',
        paid: 'Pago',
        cancelled: 'Cancelado',
        expired: 'Expirado'
    };
    return texts[status] || status;
}

function renderPagination(containerId, data, loadFunction) {
    const container = document.getElementById(containerId);
    if (!container) return;
    
    const { page, totalPages } = data;
    let html = '';
    
    // Botão anterior
    if (page > 1) {
        html += `<button onclick="${loadFunction.name}(${page - 1})">« Anterior</button>`;
    }
    
    // Números das páginas
    const startPage = Math.max(1, page - 2);
    const endPage = Math.min(totalPages, page + 2);
    
    for (let i = startPage; i <= endPage; i++) {
        html += `<button class="${i === page ? 'active' : ''}" onclick="${loadFunction.name}(${i})">${i}</button>`;
    }
    
    // Botão próximo
    if (page < totalPages) {
        html += `<button onclick="${loadFunction.name}(${page + 1})">Próximo »</button>`;
    }
    
    container.innerHTML = html;
}

function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

function showNotification(message, type = 'info') {
    // Criar elemento de notificação
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px 20px;
        border-radius: 5px;
        color: white;
        font-weight: bold;
        z-index: 10000;
        animation: slideIn 0.3s ease;
        max-width: 400px;
    `;
    
    // Cores por tipo
    const colors = {
        success: '#4CAF50',
        error: '#f44336',
        info: '#2196F3',
        warning: '#FF9800'
    };
    
    notification.style.backgroundColor = colors[type] || colors.info;
    
    document.body.appendChild(notification);
    
    // Remover após 5 segundos
    setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 5000);
}

// CSS para animações
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
`;
document.head.appendChild(style);
