// Mass Admin JavaScript

let groups = [];
let contacts = [];
let campaigns = [];
let currentPage = 1;
let currentCampaignId = null;

// Initialization
document.addEventListener('DOMContentLoaded', function() {
    loadGroups();
    loadContacts();
    loadCampaigns();
    
    // Event listeners
    document.getElementById('campaignForm').addEventListener('submit', handleCampaignSubmit);
    document.getElementById('messageText').addEventListener('input', updateMessagePreview);
    
    // Drag and drop for file upload
    setupDragAndDrop();
});

// ===== TAB NAVIGATION =====

function showTab(tabName) {
    // Hide all tabs
    document.querySelectorAll('.tab-content').forEach(tab => {
        tab.classList.remove('active');
    });
    
    // Remove active class from all buttons
    document.querySelectorAll('.tab').forEach(btn => {
        btn.classList.remove('active');
    });
    
    // Show selected tab
    document.getElementById(tabName).classList.add('active');
    
    // Activate corresponding button
    event.target.classList.add('active');
    
    // Load tab-specific data
    switch(tabName) {
        case 'groups':
            loadGroups();
            break;
        case 'contacts':
            loadContacts();
            populateGroupSelects();
            break;
        case 'campaigns':
            loadCampaigns();
            break;
        case 'create':
            populateGroupSelects();
            break;
    }
}

// ===== GRUPOS =====

async function loadGroups() {
    try {
        const token = localStorage.getItem('token');
        const response = await fetch('/api/mass-improved/groups', {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            groups = await response.json();
            renderGroups();
        }
    } catch (error) {
        console.error('Error loading groups:', error);
        showNotification('Error loading groups', 'error');
    }
}

function renderGroups() {
    const container = document.getElementById('groupsList');
    container.innerHTML = '';
    
    groups.forEach(group => {
        const groupDiv = document.createElement('div');
        groupDiv.className = 'group-card';
        groupDiv.innerHTML = `
            <div class="group-header">
                <div class="group-name">${group.group_name}</div>
                <div class="group-count">${group.contact_count} contacts</div>
            </div>
            <p>${group.description || 'No description'}</p>
            <small>Created by: ${group.created_by_name || 'System'}</small>
            <div style="margin-top: 10px;">
                <button class="btn btn-info" onclick="editGroup(${group.id})">✏️ Edit</button>
                <button class="btn btn-danger" onclick="deleteGroup(${group.id})">🗑️ Delete</button>
            </div>
        `;
        container.appendChild(groupDiv);
    });
}

async function createGroup() {
    const name = document.getElementById('groupName').value.trim();
    const description = document.getElementById('groupDescription').value.trim();
    
    if (!name) {
        showNotification('Nome do grupo é obrigatório', 'error');
        return;
    }
    
    try {
        const token = localStorage.getItem('token');
        const response = await fetch('/api/mass-improved/groups', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({ group_name: name, description })
        });
        
        if (response.ok) {
            showNotification('Grupo criado com sucesso!', 'success');
            document.getElementById('groupName').value = '';
            document.getElementById('groupDescription').value = '';
            loadGroups();
            populateGroupSelects();
        } else {
            throw new Error('Erro ao criar grupo');
        }
    } catch (error) {
        console.error('Erro ao criar grupo:', error);
        showNotification('Erro ao criar grupo', 'error');
    }
}

async function deleteGroup(groupId) {
    if (!confirm('Tem certeza que deseja excluir este grupo? Os contatos serão movidos para o grupo padrão.')) {
        return;
    }
    
    try {
        const token = localStorage.getItem('token');
        const response = await fetch(`/api/mass-improved/groups/${groupId}`, {
            method: 'DELETE',
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            showNotification('Grupo excluído com sucesso!', 'success');
            loadGroups();
            populateGroupSelects();
        } else {
            throw new Error('Erro ao excluir grupo');
        }
    } catch (error) {
        console.error('Erro ao excluir grupo:', error);
        showNotification('Erro ao excluir grupo', 'error');
    }
}

// ===== CONTATOS =====

async function loadContacts(page = 1) {
    try {
        const token = localStorage.getItem('token');
        const search = document.getElementById('contactSearch')?.value || '';
        const groupId = document.getElementById('filterGroup')?.value || '';
        
        const params = new URLSearchParams({
            page: page,
            limit: 20,
            search: search,
            group_id: groupId
        });
        
        const response = await fetch(`/api/mass-improved/contacts?${params}`, {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            const data = await response.json();
            contacts = data.contacts;
            renderContacts(data);
        }
    } catch (error) {
        console.error('Erro ao carregar contatos:', error);
        showNotification('Erro ao carregar contatos', 'error');
    }
}

function renderContacts(data) {
    const container = document.getElementById('contactsTable');
    
    if (data.contacts.length === 0) {
        container.innerHTML = '<p>Nenhum contato encontrado.</p>';
        return;
    }
    
    let html = `
        <table class="contact-table">
            <thead>
                <tr>
                    <th>Nome</th>
                    <th>Telefone</th>
                    <th>Grupo</th>
                    <th>Tags</th>
                    <th>Ações</th>
                </tr>
            </thead>
            <tbody>
    `;
    
    data.contacts.forEach(contact => {
        html += `
            <tr>
                <td>${contact.name || 'Sem nome'}</td>
                <td>${contact.phone_number}</td>
                <td>${contact.group_name || 'Sem grupo'}</td>
                <td>${contact.tags || ''}</td>
                <td>
                    <button class="btn btn-info" onclick="editContact(${contact.id})">✏️</button>
                    <button class="btn btn-danger" onclick="deleteContact(${contact.id})">🗑️</button>
                </td>
            </tr>
        `;
    });
    
    html += '</tbody></table>';
    container.innerHTML = html;
    
    // Renderizar paginação
    renderPagination('contactsPagination', data, loadContacts);
}

async function addContact() {
    const phone = document.getElementById('contactPhone').value.trim();
    const name = document.getElementById('contactName').value.trim();
    const groupId = document.getElementById('contactGroupSelect').value;
    const tags = document.getElementById('contactTags').value.trim();
    const notes = document.getElementById('contactNotes').value.trim();
    
    if (!phone) {
        showNotification('Telefone é obrigatório', 'error');
        return;
    }
    
    try {
        const token = localStorage.getItem('token');
        const response = await fetch('/api/mass-improved/contacts', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({
                phone_number: phone,
                name: name,
                group_id: groupId || null,
                tags: tags,
                notes: notes
            })
        });
        
        if (response.ok) {
            showNotification('Contato adicionado com sucesso!', 'success');
            
            // Limpar formulário
            document.getElementById('contactPhone').value = '';
            document.getElementById('contactName').value = '';
            document.getElementById('contactGroupSelect').value = '';
            document.getElementById('contactTags').value = '';
            document.getElementById('contactNotes').value = '';
            
            loadContacts();
        } else {
            const error = await response.json();
            throw new Error(error.error || 'Erro ao adicionar contato');
        }
    } catch (error) {
        console.error('Erro ao adicionar contato:', error);
        showNotification(error.message, 'error');
    }
}

async function importContacts() {
    const fileInput = document.getElementById('csvFile');
    const groupId = document.getElementById('importGroupSelect').value;
    
    if (!fileInput.files[0]) {
        showNotification('Selecione um arquivo CSV', 'error');
        return;
    }
    
    const formData = new FormData();
    formData.append('csvFile', fileInput.files[0]);
    if (groupId) {
        formData.append('group_id', groupId);
    }
    
    try {
        const token = localStorage.getItem('token');
        const response = await fetch('/api/mass-improved/contacts/import', {
            method: 'POST',
            headers: {
                'Authorization': `Bearer ${token}`
            },
            body: formData
        });
        
        if (response.ok) {
            const result = await response.json();
            showNotification(
                `Importação concluída! ${result.imported} novos, ${result.updated} atualizados. ${result.totalErrors} erros.`,
                'success'
            );
            
            fileInput.value = '';
            loadContacts();
            loadGroups();
        } else {
            const error = await response.json();
            throw new Error(error.error || 'Erro ao importar contatos');
        }
    } catch (error) {
        console.error('Erro ao importar contatos:', error);
        showNotification(error.message, 'error');
    }
}

function searchContacts() {
    clearTimeout(window.searchTimeout);
    window.searchTimeout = setTimeout(() => {
        loadContacts(1);
    }, 500);
}

function filterContacts() {
    loadContacts(1);
}

// ===== CAMPANHAS =====

async function loadCampaigns() {
    try {
        const token = localStorage.getItem('token');
        const response = await fetch('/api/mass-improved/campaigns', {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            campaigns = await response.json();
            renderCampaigns();
        }
    } catch (error) {
        console.error('Erro ao carregar campanhas:', error);
        showNotification('Erro ao carregar campanhas', 'error');
    }
}

function renderCampaigns() {
    const container = document.getElementById('campaignsList');
    container.innerHTML = '';
    
    if (campaigns.length === 0) {
        container.innerHTML = '<p>Nenhuma campanha encontrada. <a href="#" onclick="showTab(\'create\')">Criar primeira campanha</a></p>';
        return;
    }
    
    campaigns.forEach(campaign => {
        const campaignDiv = document.createElement('div');
        campaignDiv.className = 'campaign-card';
        
        const progress = campaign.total_logs > 0 ? 
            Math.round((campaign.sent_count / campaign.total_logs) * 100) : 0;
        
        campaignDiv.innerHTML = `
            <div class="campaign-header">
                <h3>${campaign.message_text.substring(0, 50)}${campaign.message_text.length > 50 ? '...' : ''}</h3>
                <span class="campaign-status status-${campaign.status}">${getStatusText(campaign.status)}</span>
            </div>
            
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-number">${campaign.total_contacts}</div>
                    <div class="stat-label">Total</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number">${campaign.sent_count || 0}</div>
                    <div class="stat-label">Enviados</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number">${campaign.failed_count || 0}</div>
                    <div class="stat-label">Falhas</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number">${progress}%</div>
                    <div class="stat-label">Progresso</div>
                </div>
            </div>
            
            <div class="progress-bar">
                <div class="progress-fill" style="width: ${progress}%"></div>
            </div>
            
            <p><strong>Grupo:</strong> ${campaign.group_name || 'Todos os contatos'}</p>
            <p><strong>Tipo:</strong> ${getMessageTypeText(campaign.message_type)}</p>
            <p><strong>Criado:</strong> ${new Date(campaign.created_at).toLocaleString('pt-BR')}</p>
            
            <div style="margin-top: 15px;">
                ${campaign.status === 'pending' ? 
                    `<button class="btn btn-success" onclick="startCampaign(${campaign.id})">🚀 Iniciar Envio</button>` : 
                    ''
                }
                <button class="btn btn-info" onclick="viewCampaignLogs(${campaign.id})">📊 Ver Logs</button>
                <button class="btn btn-warning" onclick="duplicateCampaign(${campaign.id})">📋 Duplicar</button>
            </div>
        `;
        
        container.appendChild(campaignDiv);
    });
}

async function handleCampaignSubmit(e) {
    e.preventDefault();
    
    const messageText = document.getElementById('messageText').value.trim();
    const messageType = document.getElementById('messageType').value;
    const groupId = document.getElementById('campaignGroupSelect').value;
    const sendInterval = document.getElementById('sendInterval').value;
    const scheduledAt = document.getElementById('scheduledAt').value;
    const mediaFile = document.getElementById('mediaFile').files[0];
    
    if (!messageText && !mediaFile) {
        showNotification('Mensagem ou arquivo de mídia é obrigatório', 'error');
        return;
    }
    
    const formData = new FormData();
    formData.append('message_text', messageText);
    formData.append('message_type', messageType);
    formData.append('send_interval', sendInterval);
    
    if (groupId) {
        formData.append('group_id', groupId);
    }
    
    if (scheduledAt) {
        formData.append('scheduled_at', scheduledAt);
    }
    
    if (mediaFile) {
        formData.append('mediaFile', mediaFile);
    }
    
    try {
        const token = localStorage.getItem('token');
        const response = await fetch('/api/mass-improved/campaigns', {
            method: 'POST',
            headers: {
                'Authorization': `Bearer ${token}`
            },
            body: formData
        });
        
        if (response.ok) {
            const result = await response.json();
            showNotification(`Campanha criada com sucesso! ${result.contactCount} contatos serão atingidos.`, 'success');
            
            // Limpar formulário
            document.getElementById('campaignForm').reset();
            document.getElementById('messagePreview').textContent = 'Digite uma mensagem para ver o preview...';
            
            // Voltar para aba de campanhas
            showTab('campaigns');
        } else {
            const error = await response.json();
            throw new Error(error.error || 'Erro ao criar campanha');
        }
    } catch (error) {
        console.error('Erro ao criar campanha:', error);
        showNotification(error.message, 'error');
    }
}

async function startCampaign(campaignId) {
    if (!confirm('Tem certeza que deseja iniciar esta campanha? O envio não poderá ser interrompido.')) {
        return;
    }
    
    try {
        const token = localStorage.getItem('token');
        const response = await fetch(`/api/mass-improved/campaigns/${campaignId}/start`, {
            method: 'POST',
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            const result = await response.json();
            showNotification(`Envio iniciado! ${result.contactCount} mensagens serão enviadas.`, 'success');
            loadCampaigns();
        } else {
            const error = await response.json();
            throw new Error(error.error || 'Erro ao iniciar campanha');
        }
    } catch (error) {
        console.error('Erro ao iniciar campanha:', error);
        showNotification(error.message, 'error');
    }
}

// ===== LOGS DE CAMPANHA =====

async function viewCampaignLogs(campaignId) {
    currentCampaignId = campaignId;
    document.getElementById('logsModalTitle').textContent = `Logs da Campanha #${campaignId}`;
    
    await loadCampaignStats(campaignId);
    await loadCampaignLogs(campaignId);
    
    document.getElementById('logsModal').style.display = 'block';
}

async function loadCampaignStats(campaignId) {
    try {
        const token = localStorage.getItem('token');
        const response = await fetch(`/api/mass-improved/campaigns/${campaignId}/stats`, {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            const stats = await response.json();
            renderCampaignStats(stats);
        }
    } catch (error) {
        console.error('Erro ao carregar estatísticas:', error);
    }
}

function renderCampaignStats(stats) {
    const container = document.getElementById('logsStats');
    container.innerHTML = `
        <div class="stat-card">
            <div class="stat-number">${stats.pending}</div>
            <div class="stat-label">Pendentes</div>
        </div>
        <div class="stat-card">
            <div class="stat-number">${stats.sent}</div>
            <div class="stat-label">Enviados</div>
        </div>
        <div class="stat-card">
            <div class="stat-number">${stats.failed}</div>
            <div class="stat-label">Falhas</div>
        </div>
        <div class="stat-card">
            <div class="stat-number">${stats.delivered}</div>
            <div class="stat-label">Entregues</div>
        </div>
    `;
}

async function loadCampaignLogs(campaignId, page = 1) {
    try {
        const token = localStorage.getItem('token');
        const status = document.getElementById('logsStatusFilter')?.value || '';
        
        const params = new URLSearchParams({
            page: page,
            limit: 20,
            status: status
        });
        
        const response = await fetch(`/api/mass-improved/campaigns/${campaignId}/logs?${params}`, {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            const data = await response.json();
            renderCampaignLogs(data);
        }
    } catch (error) {
        console.error('Erro ao carregar logs:', error);
    }
}

function renderCampaignLogs(data) {
    const container = document.getElementById('logsList');
    container.innerHTML = '';
    
    data.logs.forEach(log => {
        const logDiv = document.createElement('div');
        logDiv.className = 'log-item';
        logDiv.innerHTML = `
            <div>
                <div class="log-contact">${log.contact_name || log.contact_phone}</div>
                <small>${log.contact_phone}</small>
            </div>
            <div>
                <span class="log-status log-${log.status}">${getStatusText(log.status)}</span>
                <br>
                <small>${new Date(log.created_at).toLocaleString('pt-BR')}</small>
            </div>
        `;
        container.appendChild(logDiv);
    });
    
    // Renderizar paginação
    renderPagination('logsPagination', data, (page) => loadCampaignLogs(currentCampaignId, page));
}

function closeLogsModal() {
    document.getElementById('logsModal').style.display = 'none';
    currentCampaignId = null;
}

function filterLogs() {
    if (currentCampaignId) {
        loadCampaignLogs(currentCampaignId, 1);
    }
}

function refreshLogs() {
    if (currentCampaignId) {
        loadCampaignStats(currentCampaignId);
        loadCampaignLogs(currentCampaignId, 1);
    }
}

// ===== UTILITÁRIOS =====

function populateGroupSelects() {
    const selects = [
        'importGroupSelect',
        'contactGroupSelect', 
        'campaignGroupSelect',
        'filterGroup'
    ];
    
    selects.forEach(selectId => {
        const select = document.getElementById(selectId);
        if (select) {
            // Manter opção padrão
            const defaultOption = select.querySelector('option[value=""]');
            select.innerHTML = '';
            if (defaultOption) {
                select.appendChild(defaultOption);
            }
            
            groups.forEach(group => {
                const option = document.createElement('option');
                option.value = group.id;
                option.textContent = group.group_name;
                select.appendChild(option);
            });
        }
    });
}

function toggleMediaUpload() {
    const messageType = document.getElementById('messageType').value;
    const mediaGroup = document.getElementById('mediaUploadGroup');
    
    if (messageType === 'text') {
        mediaGroup.style.display = 'none';
    } else {
        mediaGroup.style.display = 'block';
        updateMediaUploadText();
    }
}

function updateMediaUploadText() {
    const messageType = document.getElementById('messageType').value;
    const uploadText = document.getElementById('mediaUploadText');
    
    const texts = {
        image: '🖼️ Clique para selecionar imagem',
        video: '🎥 Clique para selecionar vídeo',
        audio: '🎵 Clique para selecionar áudio',
        document: '📄 Clique para selecionar documento'
    };
    
    uploadText.textContent = texts[messageType] || '📁 Clique para selecionar arquivo';
}

function handleFileSelect(input) {
    const file = input.files[0];
    if (file) {
        const uploadText = document.getElementById('mediaUploadText') || 
                          input.parentElement.querySelector('p');
        uploadText.textContent = `📁 ${file.name}`;
    }
}

function handleMediaSelect(input) {
    handleFileSelect(input);
    updateMessagePreview();
}

function updateMessagePreview() {
    const messageText = document.getElementById('messageText').value;
    const mediaFile = document.getElementById('mediaFile').files[0];
    const preview = document.getElementById('messagePreview');
    
    let previewHtml = '';
    
    if (mediaFile) {
        const fileType = mediaFile.type.split('/')[0];
        if (fileType === 'image') {
            const url = URL.createObjectURL(mediaFile);
            previewHtml += `<img src="${url}" class="preview-media" alt="Preview">`;
        } else {
            previewHtml += `<div>📎 ${mediaFile.name}</div>`;
        }
    }
    
    if (messageText) {
        previewHtml += `<div>${processPlaceholdersForPreview(messageText)}</div>`;
    }
    
    preview.innerHTML = previewHtml || 'Digite uma mensagem para ver o preview...';
}

function processPlaceholdersForPreview(text) {
    let processed = text;
    
    // Substituir placeholders comuns para preview
    processed = processed.replace(/\{\{nome_cliente\}\}/g, 'João Silva');
    processed = processed.replace(/\{\{nome_loja\}\}/g, 'AI Moda');
    processed = processed.replace(/\{\{horario_funcionamento\}\}/g, '08:00 às 18:00');
    processed = processed.replace(/\{\{telefone_loja\}\}/g, '(11) 99999-9999');
    processed = processed.replace(/\{\{endereco_loja\}\}/g, 'Rua das Flores, 123');
    processed = processed.replace(/\{\{data_atual\}\}/g, new Date().toLocaleDateString('pt-BR'));
    processed = processed.replace(/\{\{hora_atual\}\}/g, new Date().toLocaleTimeString('pt-BR'));
    
    return processed;
}

function setupDragAndDrop() {
    const dropZones = document.querySelectorAll('.file-upload');
    
    dropZones.forEach(zone => {
        zone.addEventListener('dragover', (e) => {
            e.preventDefault();
            zone.classList.add('dragover');
        });
        
        zone.addEventListener('dragleave', () => {
            zone.classList.remove('dragover');
        });
        
        zone.addEventListener('drop', (e) => {
            e.preventDefault();
            zone.classList.remove('dragover');
            
            const files = e.dataTransfer.files;
            if (files.length > 0) {
                const input = zone.querySelector('input[type="file"]');
                if (input) {
                    input.files = files;
                    handleFileSelect(input);
                }
            }
        });
    });
}

function renderPagination(containerId, data, loadFunction) {
    const container = document.getElementById(containerId);
    if (!container) return;
    
    const { page, totalPages } = data;
    let html = '';
    
    // Botão anterior
    if (page > 1) {
        html += `<button onclick="${loadFunction.name}(${page - 1})">« Anterior</button>`;
    }
    
    // Números das páginas
    const startPage = Math.max(1, page - 2);
    const endPage = Math.min(totalPages, page + 2);
    
    for (let i = startPage; i <= endPage; i++) {
        html += `<button class="${i === page ? 'active' : ''}" onclick="${loadFunction.name}(${i})">${i}</button>`;
    }
    
    // Botão próximo
    if (page < totalPages) {
        html += `<button onclick="${loadFunction.name}(${page + 1})">Próximo »</button>`;
    }
    
    container.innerHTML = html;
}

function getStatusText(status) {
    const texts = {
        pending: 'Pendente',
        sending: 'Enviando',
        completed: 'Concluído',
        failed: 'Falhou',
        sent: 'Enviado',
        delivered: 'Entregue'
    };
    return texts[status] || status;
}

function getMessageTypeText(type) {
    const texts = {
        text: '📝 Texto',
        image: '🖼️ Imagem',
        video: '🎥 Vídeo',
        audio: '🎵 Áudio',
        document: '📄 Documento'
    };
    return texts[type] || type;
}

function showNotification(message, type = 'info') {
    // Criar elemento de notificação
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px 20px;
        border-radius: 5px;
        color: white;
        font-weight: bold;
        z-index: 10000;
        animation: slideIn 0.3s ease;
        max-width: 400px;
    `;
    
    // Cores por tipo
    const colors = {
        success: '#4CAF50',
        error: '#f44336',
        info: '#2196F3',
        warning: '#FF9800'
    };
    
    notification.style.backgroundColor = colors[type] || colors.info;
    
    document.body.appendChild(notification);
    
    // Remover após 5 segundos
    setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 5000);
}

// CSS para animações
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
`;
document.head.appendChild(style);
