// API Configuration
const API_BASE = window.location.origin;

// Elementos DOM
const loginForm = document.getElementById('loginForm');
const errorMessage = document.getElementById('errorMessage');

// Event Listeners
document.addEventListener('DOMContentLoaded', function() {
    checkAuthStatus();
    if (loginForm) {
        loginForm.addEventListener('submit', handleLogin);
    }
});

// SERVICE WORKER REGISTRATION
if ('serviceWorker' in navigator) {
  window.addEventListener('load', () => {
    navigator.serviceWorker.register('/service-worker.js')
      .then(registration => {
        console.log('Service Worker registered successfully:', registration);
      })
      .catch(error => {
        console.error('Failed to register Service Worker:', error);
      });
  });
}

// Check authentication status - WITHOUT AUTOMATIC REDIRECTS
async function checkAuthStatus() {
    const token = localStorage.getItem('authToken');
    const currentPath = window.location.pathname;
    
    // If there's no token, just return without redirecting
    if (!token) {
        console.log('User not authenticated');
        return false;
    }
    
    try {
        const response = await fetch(`${API_BASE}/api/auth/verify`, {
            headers: { 'Authorization': `Bearer ${token}` }
        });
        
        if (response.ok) {
            const data = await response.json();
            // Handle response structure: { success: true, data: { valid, user } } or { valid, user }
            const responseData = data.data || data;
            const isValid = responseData.valid !== undefined ? responseData.valid : data.valid;
            const user = responseData.user || data.user;
            
            if (isValid && user) {
                console.log('User authenticated:', user);
                
                // Check permissions only for admin pages
                if (currentPath.includes('/admin') && user.role !== 'admin') {
                    showError('Access denied. You do not have permission to access this page.');
                    setTimeout(() => {
                        window.location.href = '/user';
                    }, 2000);
                    return false;
                }
                
                // If on login page and already authenticated, redirect
                if (currentPath === '/' || currentPath === '/index.html' || currentPath === '/login') {
                    redirectToDashboard(user.role);
                    return true;
                }
                
                return user;
            }
        } else {
            clearAuthData();
            return false;
        }
    } catch (error) {
        console.error('Error verifying authentication:', error);
        clearAuthData();
        return false;
    }
}

// Handle login
async function handleLogin(e) {
    e.preventDefault();
    const username = document.getElementById('username').value;
    const password = document.getElementById('password').value;
    
    if (!username || !password) {
        showError('Please fill in all fields');
        return;
    }

    // Show loading
    const submitBtn = e.target.querySelector('button[type="submit"]');
    const originalText = submitBtn.textContent;
    submitBtn.textContent = 'Logging in...';
    submitBtn.disabled = true;
    
    try {
        const response = await fetch(`${API_BASE}/api/auth/login`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ username, password })
        });
        
        const data = await response.json();
        
        if (response.ok && data.success) {
            // Extract data from response structure: { success: true, data: { token, user } }
            const responseData = data.data || data;
            const token = responseData.token;
            const user = responseData.user;
            
            if (!token || !user) {
                showError('Invalid response format');
                return;
            }
            
            // Save token and user data
            localStorage.setItem('authToken', token);
            localStorage.setItem('currentUser', JSON.stringify(user));
            
            showNotification('Login successful!', 'success');
            redirectToDashboard(user.role);
        } else {
            showError(data.error || data.message || 'Login error');
        }
    } catch (error) {
        console.error('Login error:', error);
        showError('Connection error. Please try again.');
    } finally {
        // Restore button
        submitBtn.textContent = originalText;
        submitBtn.disabled = false;
    }
}

// Redirect to appropriate dashboard
function redirectToDashboard(role) {
    if (role === 'admin') {
        window.location.href = '/admin';
    } else {
        window.location.href = '/user';
    }
}

// Show error message
function showError(message) {
    if (errorMessage) {
        errorMessage.textContent = message;
        errorMessage.style.display = 'block';
        errorMessage.classList.add('show');
        setTimeout(() => {
            errorMessage.style.display = 'none';
            errorMessage.classList.remove('show');
        }, 5000);
    } else {
        showNotification(message, 'error');
    }
}

// Clear authentication data
function clearAuthData() {
    localStorage.removeItem('authToken');
    localStorage.removeItem('currentUser');
}

// Logout
function logout() {
    clearAuthData();
    showNotification('Logout successful!', 'success');
    setTimeout(() => {
        window.location.href = '/';
    }, 1000);
}

// Get user data
function getCurrentUser() {
    const userStr = localStorage.getItem('currentUser');
    return userStr ? JSON.parse(userStr) : null;
}

// Make authenticated request
async function authenticatedFetch(url, options = {}) {
    const token = localStorage.getItem('authToken');
    
    if (!token) {
        logout();
        throw new Error('Token not found');
    }
    
    const headers = {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`,
        ...options.headers
    };
    
    const response = await fetch(url, { ...options, headers });
    
    if (response.status === 401 || response.status === 403) {
        logout();
        throw new Error('Session expired');
    }
    
    return response;
}

// Utility functions
function showNotification(message, type = 'info') {
    // Remove previous notifications
    const existingNotifications = document.querySelectorAll('.notification');
    existingNotifications.forEach(n => n.remove());

    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.innerHTML = `
        <span>${message}</span>
        <button onclick="this.parentElement.remove()" style="margin-left: 10px; background: none; border: none; color: inherit; cursor: pointer;">&times;</button>
    `;
    
    // Inline styles to ensure it works
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px 20px;
        border-radius: 5px;
        color: white;
        font-weight: bold;
        z-index: 10000;
        max-width: 300px;
        box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        animation: slideIn 0.3s ease;
    `;

    // Colors based on type
    const colors = {
        success: '#28a745',
        error: '#dc3545',
        warning: '#ffc107',
        info: '#17a2b8'
    };
    notification.style.backgroundColor = colors[type] || colors.info;

    document.body.appendChild(notification);
    
    setTimeout(() => {
        if (notification.parentElement) {
            notification.remove();
        }
    }, 5000);
}

function formatDate(dateString) {
    if (!dateString) return '';
    const date = new Date(dateString);
    return date.toLocaleString('en-US', {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    });
}

function formatPhone(phone) {
    if (!phone) return 'N/A';
    const cleaned = phone.replace(/\D/g, '');
    if (cleaned.length === 13) return `+${cleaned.slice(0, 2)} (${cleaned.slice(2, 4)}) ${cleaned.slice(4, 9)}-${cleaned.slice(9)}`;
    if (cleaned.length === 11) return `(${cleaned.slice(0, 2)}) ${cleaned.slice(2, 7)}-${cleaned.slice(7)}`;
    if (cleaned.length === 10) return `(${cleaned.slice(0, 2)}) ${cleaned.slice(2, 6)}-${cleaned.slice(6)}`;
    return phone;
}

// CSS para animações (injetado via JavaScript)
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from {
            transform: translateX(100%);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
    
    .notification {
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    }
`;
document.head.appendChild(style);
