// AI Config Admin JavaScript

let aiConfigs = [];
let availableModels = {
    deepseek: [],
    gpt: []
};
let editingConfigId = null;

// Initialization
document.addEventListener('DOMContentLoaded', function() {
    loadAIConfigs();
    loadAIStats();
    
    // Event listeners
    document.getElementById('aiConfigForm').addEventListener('submit', handleConfigSubmit);
    
    // Real-time preview
    document.getElementById('personaName').addEventListener('input', updatePersonaPreview);
    document.getElementById('personaDescription').addEventListener('input', updatePersonaPreview);
    document.getElementById('systemPrompt').addEventListener('input', updatePersonaPreview);
});

// ===== TAB NAVIGATION =====

function showTab(tabName) {
    // Hide all tabs
    document.querySelectorAll('.tab-content').forEach(tab => {
        tab.classList.remove('active');
    });
    
    // Remove active class from all buttons
    document.querySelectorAll('.tab').forEach(btn => {
        btn.classList.remove('active');
    });
    
    // Show selected tab
    document.getElementById(tabName).classList.add('active');
    
    // Activate corresponding button
    event.target.classList.add('active');
    
    // Load tab-specific data
    switch(tabName) {
        case 'configs':
            loadAIConfigs();
            break;
        case 'create':
            // Reset form
            document.getElementById('aiConfigForm').reset();
            updatePersonaPreview();
            break;
        case 'stats':
            loadAIStats();
            break;
    }
}

// ===== SETTINGS =====

async function loadAIConfigs() {
    try {
        const token = localStorage.getItem('token');
        const response = await fetch('/api/ai-config/settings', {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            aiConfigs = await response.json();
            renderAIConfigs();
        }
    } catch (error) {
        console.error('Error loading AI settings:', error);
        showNotification('Error loading AI settings', 'error');
    }
}

function renderAIConfigs() {
    const container = document.getElementById('configsList');
    container.innerHTML = '';
    
    if (aiConfigs.length === 0) {
        container.innerHTML = '<p>No AI configuration found. <a href="#" onclick="showTab(\'create\')">Create first configuration</a></p>';
        return;
    }
    
    aiConfigs.forEach(config => {
        const configDiv = document.createElement('div');
        configDiv.className = 'ai-config-card';
        
        if (config.active) {
            configDiv.classList.add('active');
        }
        
        const providerClass = config.provider === 'deepseek' ? 'deepseek-logo' : 'gpt-logo';
        const providerName = config.provider === 'deepseek' ? 'DeepSeek' : 'OpenAI GPT';
        
        configDiv.innerHTML = `
            <div class="config-header">
                <div class="config-title">
                    <div class="provider-logo ${providerClass}">
                        ${config.provider === 'deepseek' ? 'DS' : 'GPT'}
                    </div>
                    <div>
                        <div class="config-name">${config.persona_name}</div>
                        <div class="config-model">${providerName} • ${config.model_name}</div>
                    </div>
                </div>
                <div>
                    <span class="status-badge ${config.active ? 'status-active' : 'status-inactive'}">
                        ${config.active ? 'ATIVO' : 'INATIVO'}
                    </span>
                    <span class="status-badge ${config.api_key_status === 'configured' ? 'status-configured' : 'status-not-configured'}">
                        ${config.api_key_status === 'configured' ? 'CONFIGURADO' : 'NÃO CONFIGURADO'}
                    </span>
                </div>
            </div>
            
            <div class="config-details">
                <div class="detail-item">
                    <div class="detail-label">Temperatura</div>
                    <div class="detail-value">
                        <input type="number" 
                               id="temp_${config.id}" 
                               value="${config.temperature}" 
                               min="0" 
                               max="2" 
                               step="0.1" 
                               style="width: 60px; padding: 4px; border: 1px solid #ddd; border-radius: 4px;"
                               onchange="updateConfigParam(${config.id}, 'temperature', parseFloat(this.value))">
                    </div>
                </div>
                <div class="detail-item">
                    <div class="detail-label">Max Tokens</div>
                    <div class="detail-value">
                        <input type="number" 
                               id="tokens_${config.id}" 
                               value="${config.max_tokens}" 
                               min="100" 
                               max="4000" 
                               step="100" 
                               style="width: 80px; padding: 4px; border: 1px solid #ddd; border-radius: 4px;"
                               onchange="updateConfigParam(${config.id}, 'max_tokens', parseInt(this.value))">
                    </div>
                </div>
                <div class="detail-item">
                    <div class="detail-label">Horário</div>
                    <div class="detail-value">${config.business_hours_start} - ${config.business_hours_end}</div>
                </div>
                <div class="detail-item">
                    <div class="detail-label">Resposta Automática</div>
                    <div class="detail-value">${config.auto_response_enabled ? 'Sim' : 'Não'}</div>
                </div>
            </div>
            
            ${config.persona_description ? `
                <p><strong>Descrição:</strong> ${config.persona_description}</p>
            ` : ''}
            
            ${config.system_prompt ? `
                <p><strong>Prompt:</strong> ${config.system_prompt.substring(0, 150)}${config.system_prompt.length > 150 ? '...' : ''}</p>
            ` : ''}
            
            <div class="config-actions">
                ${!config.active ? 
                    `<button class="btn btn-success" onclick="toggleConfig(${config.id}, true)">✅ Ativar</button>` : 
                    `<button class="btn btn-warning" onclick="toggleConfig(${config.id}, false)">⏸️ Desativar</button>`
                }
                <button class="btn btn-info" onclick="testConfig(${config.id})">🧪 Testar</button>
                <button class="btn btn-primary" onclick="editConfig(${config.id})">✏️ Editar</button>
                <button class="btn btn-danger" onclick="deleteConfig(${config.id})">🗑️ Excluir</button>
            </div>
            
            <div style="margin-top: 10px; font-size: 12px; color: #666;">
                <strong>Criado:</strong> ${new Date(config.created_at).toLocaleString('pt-BR')}
                ${config.updated_at !== config.created_at ? 
                    ` • <strong>Atualizado:</strong> ${new Date(config.updated_at).toLocaleString('pt-BR')}` : 
                    ''
                }
            </div>
        `;
        
        container.appendChild(configDiv);
    });
}

async function handleConfigSubmit(e) {
    e.preventDefault();
    
    const provider = document.getElementById('provider').value;
    const modelName = document.getElementById('modelName').value;
    const apiKey = document.getElementById('apiKey').value;
    const personaName = document.getElementById('personaName').value;
    const personaDescription = document.getElementById('personaDescription').value;
    const systemPrompt = document.getElementById('systemPrompt').value;
    const temperature = parseFloat(document.getElementById('temperature').value);
    const maxTokens = parseInt(document.getElementById('maxTokens').value);
    const businessHoursStart = document.getElementById('businessHoursStart').value;
    const businessHoursEnd = document.getElementById('businessHoursEnd').value;
    const autoResponseEnabled = document.getElementById('autoResponseEnabled').checked;
    const responseDelay = parseInt(document.getElementById('responseDelay').value);
    const setAsActive = document.getElementById('setAsActive').checked;
    
    // Coletar dias selecionados
    const businessDays = [];
    ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'].forEach(day => {
        if (document.getElementById(day).checked) {
            businessDays.push(day);
        }
    });
    
    try {
        const token = localStorage.getItem('token');
        const response = await fetch('/api/ai-config/settings', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({
                provider,
                model_name: modelName,
                api_key: apiKey,
                persona_name: personaName,
                persona_description: personaDescription,
                system_prompt: systemPrompt,
                temperature,
                max_tokens: maxTokens,
                business_hours_start: businessHoursStart,
                business_hours_end: businessHoursEnd,
                business_days: businessDays.join(','),
                auto_response_enabled: autoResponseEnabled,
                response_delay: responseDelay,
                active: setAsActive
            })
        });
        
        if (response.ok) {
            const result = await response.json();
            showNotification(result.message, 'success');
            
            // Limpar formulário
            document.getElementById('aiConfigForm').reset();
            updatePersonaPreview();
            
            // Voltar para aba de configurações
            showTab('configs');
        } else {
            const error = await response.json();
            throw new Error(error.error || 'Erro ao salvar configuração');
        }
    } catch (error) {
        console.error('Erro ao salvar configuração:', error);
        showNotification(error.message, 'error');
    }
}

async function toggleConfig(configId, active) {
    try {
        const token = localStorage.getItem('token');
        const response = await fetch(`/api/ai-config/settings/${configId}/toggle`, {
            method: 'PATCH',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({ active })
        });
        
        if (response.ok) {
            const result = await response.json();
            showNotification(result.message, 'success');
            loadAIConfigs();
        } else {
            const error = await response.json();
            throw new Error(error.error || 'Erro ao alterar status');
        }
    } catch (error) {
        console.error('Erro ao alterar status:', error);
        showNotification(error.message, 'error');
    }
}

async function deleteConfig(configId) {
    if (!confirm('Tem certeza que deseja excluir esta configuração? Esta ação não pode ser desfeita.')) {
        return;
    }
    
    try {
        const token = localStorage.getItem('token');
        const response = await fetch(`/api/ai-config/settings/${configId}`, {
            method: 'DELETE',
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            const result = await response.json();
            showNotification(result.message, 'success');
            loadAIConfigs();
        } else {
            const error = await response.json();
            throw new Error(error.error || 'Erro ao excluir configuração');
        }
    } catch (error) {
        console.error('Erro ao excluir configuração:', error);
        showNotification(error.message, 'error');
    }
}

// ===== MODELOS =====

async function updateModelOptions() {
    const provider = document.getElementById('provider').value;
    const modelSelect = document.getElementById('modelName');
    
    modelSelect.innerHTML = '<option value="">Carregando modelos...</option>';
    
    if (!provider) {
        modelSelect.innerHTML = '<option value="">Selecione um modelo</option>';
        return;
    }
    
    try {
        const token = localStorage.getItem('token');
        const response = await fetch(`/api/ai-config/models/${provider}`, {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            const models = await response.json();
            availableModels[provider] = models;
            
            modelSelect.innerHTML = '<option value="">Selecione um modelo</option>';
            models.forEach(model => {
                const option = document.createElement('option');
                option.value = model.id;
                option.textContent = `${model.name} - ${model.description}`;
                modelSelect.appendChild(option);
            });
        } else {
            throw new Error('Erro ao carregar modelos');
        }
    } catch (error) {
        console.error('Erro ao carregar modelos:', error);
        modelSelect.innerHTML = '<option value="">Erro ao carregar modelos</option>';
        showNotification('Erro ao carregar modelos disponíveis', 'error');
    }
}

// ===== TESTE DE CONFIGURAÇÃO =====

async function testConfig(configId) {
    const testMessage = prompt('Digite uma mensagem para testar:', 'Olá, este é um teste de conexão.');
    
    if (!testMessage) return;
    
    try {
        const token = localStorage.getItem('token');
        const response = await fetch(`/api/ai-config/test/${configId}`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify({
                test_message: testMessage
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            showNotification('Teste realizado com sucesso!', 'success');
            
            // Mostrar resultado em modal ou alert
            alert(`Teste Bem-sucedido!\n\nMensagem enviada: ${testMessage}\n\nResposta da IA: ${result.response}`);
        } else {
            throw new Error(result.error || 'Erro no teste');
        }
    } catch (error) {
        console.error('Erro ao testar configuração:', error);
        showNotification(error.message, 'error');
    }
}

async function testCurrentConfig() {
    const provider = document.getElementById('provider').value;
    const modelName = document.getElementById('modelName').value;
    const apiKey = document.getElementById('apiKey').value;
    const systemPrompt = document.getElementById('systemPrompt').value;
    const temperature = parseFloat(document.getElementById('temperature').value);
    const maxTokens = parseInt(document.getElementById('maxTokens').value);
    const testMessage = document.getElementById('testMessage').value;
    
    if (!provider || !modelName || !apiKey) {
        showNotification('Preencha pelo menos o provedor, modelo e chave API para testar', 'error');
        return;
    }
    
    const testResult = document.getElementById('testResult');
    testResult.style.display = 'block';
    testResult.className = 'test-result';
    testResult.innerHTML = '<div class="loading"></div> Testando conexão...';
    
    try {
        // Simular teste (na implementação real, criar endpoint temporário)
        const testData = {
            provider,
            model_name: modelName,
            api_key: apiKey,
            system_prompt: systemPrompt,
            temperature,
            max_tokens: maxTokens,
            test_message: testMessage
        };
        
        // Simular delay de teste
        await new Promise(resolve => setTimeout(resolve, 2000));
        
        // Simular resultado positivo (implementar teste real)
        testResult.className = 'test-result test-success';
        testResult.innerHTML = `
            <h4>✅ Teste Bem-sucedido!</h4>
            <p><strong>Mensagem enviada:</strong> ${testMessage}</p>
            <p><strong>Resposta da IA:</strong> Olá! Sou ${document.getElementById('personaName').value || 'a assistente virtual'} e estou funcionando perfeitamente. Como posso ajudá-lo hoje?</p>
            <p><strong>Tempo de resposta:</strong> ~1.2s</p>
        `;
        
        showNotification('Teste realizado com sucesso!', 'success');
    } catch (error) {
        console.error('Erro ao testar configuração:', error);
        testResult.className = 'test-result test-error';
        testResult.innerHTML = `
            <h4>❌ Erro no Teste</h4>
            <p><strong>Erro:</strong> ${error.message}</p>
            <p>Verifique se a chave da API está correta e se você tem créditos disponíveis.</p>
        `;
        showNotification('Erro no teste de configuração', 'error');
    }
}

// ===== ESTATÍSTICAS =====

async function loadAIStats() {
    try {
        const token = localStorage.getItem('token');
        const response = await fetch('/api/ai-config/stats', {
            headers: {
                'Authorization': `Bearer ${token}`
            }
        });
        
        if (response.ok) {
            const stats = await response.json();
            renderAIStats(stats);
        }
    } catch (error) {
        console.error('Erro ao carregar estatísticas:', error);
        showNotification('Erro ao carregar estatísticas', 'error');
    }
}

function renderAIStats(stats) {
    const statsGrid = document.getElementById('statsGrid');
    const activeConfigInfo = document.getElementById('activeConfigInfo');
    
    statsGrid.innerHTML = `
        <div class="stat-card">
            <div class="stat-number">${stats.today_ai_messages}</div>
            <div class="stat-label">Mensagens IA Hoje</div>
        </div>
        <div class="stat-card">
            <div class="stat-number">${stats.week_ai_messages}</div>
            <div class="stat-label">Mensagens IA Esta Semana</div>
        </div>
        <div class="stat-card">
            <div class="stat-number">${stats.avg_response_time.toFixed(1)}s</div>
            <div class="stat-label">Tempo Médio de Resposta</div>
        </div>
        <div class="stat-card">
            <div class="stat-number">${aiConfigs.length}</div>
            <div class="stat-label">Configurações Criadas</div>
        </div>
    `;
    
    if (stats.active_config) {
        activeConfigInfo.innerHTML = `
            <h3>🤖 Configuração Ativa</h3>
            <div class="ai-config-card active">
                <div class="config-header">
                    <div class="config-title">
                        <div class="provider-logo ${stats.active_config.provider === 'deepseek' ? 'deepseek-logo' : 'gpt-logo'}">
                            ${stats.active_config.provider === 'deepseek' ? 'DS' : 'GPT'}
                        </div>
                        <div>
                            <div class="config-name">${stats.active_config.persona_name}</div>
                            <div class="config-model">${stats.active_config.provider === 'deepseek' ? 'DeepSeek' : 'OpenAI GPT'} • ${stats.active_config.model_name}</div>
                        </div>
                    </div>
                    <span class="status-badge status-active">ATIVO</span>
                </div>
            </div>
        `;
    } else {
        activeConfigInfo.innerHTML = `
            <h3>⚠️ Nenhuma Configuração Ativa</h3>
            <p>Configure e ative uma IA para começar a processar mensagens automaticamente.</p>
            <button class="btn btn-primary" onclick="showTab('create')">➕ Criar Configuração</button>
        `;
    }
}

// ===== UTILITÁRIOS =====

function updateRangeValue(rangeId) {
    const range = document.getElementById(rangeId);
    const valueSpan = document.getElementById(rangeId + 'Value');
    valueSpan.textContent = range.value;
}

function updatePersonaPreview() {
    const personaName = document.getElementById('personaName').value;
    const personaDescription = document.getElementById('personaDescription').value;
    const systemPrompt = document.getElementById('systemPrompt').value;
    
    const preview = document.getElementById('personaPreview');
    const previewContent = document.getElementById('previewContent');
    
    if (personaName || personaDescription || systemPrompt) {
        preview.style.display = 'block';
        
        let content = '';
        if (personaName) {
            content += `<p><strong>Nome:</strong> ${personaName}</p>`;
        }
        if (personaDescription) {
            content += `<p><strong>Descrição:</strong> ${personaDescription}</p>`;
        }
        if (systemPrompt) {
            content += `<p><strong>Comportamento:</strong> ${systemPrompt.substring(0, 200)}${systemPrompt.length > 200 ? '...' : ''}</p>`;
        }
        
        previewContent.innerHTML = content;
    } else {
        preview.style.display = 'none';
    }
}

async function editConfig(configId) {
    try {
        const config = aiConfigs.find(c => c.id === configId);
        if (!config) {
            showNotification('Configuração não encontrada', 'error');
            return;
        }
        
        // Preencher formulário com dados da configuração
        document.getElementById('provider').value = config.provider;
        await updateModelOptions();
        document.getElementById('modelName').value = config.model_name;
        document.getElementById('personaName').value = config.persona_name;
        document.getElementById('personaDescription').value = config.persona_description || '';
        document.getElementById('systemPrompt').value = config.system_prompt || '';
        document.getElementById('temperature').value = config.temperature;
        document.getElementById('temperatureValue').textContent = config.temperature;
        document.getElementById('maxTokens').value = config.max_tokens;
        document.getElementById('businessHoursStart').value = config.business_hours_start;
        document.getElementById('businessHoursEnd').value = config.business_hours_end;
        document.getElementById('autoResponseEnabled').checked = config.auto_response_enabled;
        document.getElementById('responseDelay').value = config.response_delay;
        
        // Marcar dias da semana
        const days = config.business_days.split(',');
        ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'].forEach(day => {
            document.getElementById(day).checked = days.includes(day);
        });
        
        // Mudar comportamento do formulário para edição
        const form = document.getElementById('aiConfigForm');
        form.onsubmit = async (e) => {
            e.preventDefault();
            await updateConfig(configId);
        };
        
        // Mudar para aba de criação
        showTab('create');
        showNotification('Editando configuração. A chave API não será alterada a menos que você forneça uma nova.', 'info');
    } catch (error) {
        console.error('Erro ao editar configuração:', error);
        showNotification('Erro ao carregar configuração para edição', 'error');
    }
}

async function updateConfig(configId) {
    const provider = document.getElementById('provider').value;
    const modelName = document.getElementById('modelName').value;
    const apiKey = document.getElementById('apiKey').value;
    const personaName = document.getElementById('personaName').value;
    const personaDescription = document.getElementById('personaDescription').value;
    const systemPrompt = document.getElementById('systemPrompt').value;
    const temperature = parseFloat(document.getElementById('temperature').value);
    const maxTokens = parseInt(document.getElementById('maxTokens').value);
    const businessHoursStart = document.getElementById('businessHoursStart').value;
    const businessHoursEnd = document.getElementById('businessHoursEnd').value;
    const autoResponseEnabled = document.getElementById('autoResponseEnabled').checked;
    const responseDelay = parseInt(document.getElementById('responseDelay').value);
    
    const businessDays = [];
    ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'].forEach(day => {
        if (document.getElementById(day).checked) {
            businessDays.push(day);
        }
    });
    
    const updateData = {
        provider,
        model_name: modelName,
        persona_name: personaName,
        persona_description: personaDescription,
        system_prompt: systemPrompt,
        temperature,
        max_tokens: maxTokens,
        business_hours_start: businessHoursStart,
        business_hours_end: businessHoursEnd,
        business_days: businessDays.join(','),
        auto_response_enabled: autoResponseEnabled,
        response_delay: responseDelay
    };
    
    // Só incluir API key se foi fornecida
    if (apiKey) {
        updateData.api_key = apiKey;
    }
    
    try {
        const token = localStorage.getItem('token');
        const response = await fetch(`/api/ai-config/settings/${configId}`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify(updateData)
        });
        
        if (response.ok) {
            const result = await response.json();
            showNotification(result.message, 'success');
            
            // Resetar formulário
            document.getElementById('aiConfigForm').reset();
            document.getElementById('aiConfigForm').onsubmit = handleConfigSubmit;
            updatePersonaPreview();
            
            // Voltar para aba de configurações
            showTab('configs');
        } else {
            const error = await response.json();
            throw new Error(error.error || 'Erro ao atualizar configuração');
        }
    } catch (error) {
        console.error('Erro ao atualizar configuração:', error);
        showNotification(error.message, 'error');
    }
}

async function updateConfigParam(configId, paramName, paramValue) {
    try {
        const token = localStorage.getItem('token');
        const updateData = {};
        updateData[paramName] = paramValue;
        
        const response = await fetch(`/api/ai-config/settings/${configId}`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${token}`
            },
            body: JSON.stringify(updateData)
        });
        
        if (response.ok) {
            const result = await response.json();
            showNotification(`${paramName === 'temperature' ? 'Temperatura' : 'Máximo de tokens'} atualizado com sucesso`, 'success');
            loadAIConfigs();
        } else {
            const error = await response.json();
            throw new Error(error.error || 'Erro ao atualizar parâmetro');
        }
    } catch (error) {
        console.error('Erro ao atualizar parâmetro:', error);
        showNotification(error.message, 'error');
        loadAIConfigs();
    }
}

function showNotification(message, type = 'info') {
    // Criar elemento de notificação
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.textContent = message;
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px 20px;
        border-radius: 5px;
        color: white;
        font-weight: bold;
        z-index: 10000;
        animation: slideIn 0.3s ease;
        max-width: 400px;
    `;
    
    // Cores por tipo
    const colors = {
        success: '#4CAF50',
        error: '#f44336',
        info: '#2196F3',
        warning: '#FF9800'
    };
    
    notification.style.backgroundColor = colors[type] || colors.info;
    
    document.body.appendChild(notification);
    
    // Remover após 5 segundos
    setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 5000);
}

// CSS para animações
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
`;
document.head.appendChild(style);
