/**
 * CSRF Protection Middleware
 * 
 * Simple CSRF token implementation for HTML forms
 * Note: For REST APIs with JWT, CSRF is less critical but still recommended
 */

const crypto = require('crypto');

// Store tokens in memory (in production, use Redis or session store)
const csrfTokens = new Map();

/**
 * Generate a CSRF token
 * @returns {string} CSRF token
 */
function generateToken() {
  return crypto.randomBytes(32).toString('hex');
}

/**
 * Middleware to generate and attach CSRF token to response
 * Token is sent in response header and should be included in subsequent requests
 */
function generateCSRFToken(req, res, next) {
  // Only generate token for HTML pages (not API endpoints)
  if (req.path.startsWith('/api/')) {
    return next();
  }

  // Generate token
  const token = generateToken();
  
  // Store token with expiration (15 minutes)
  csrfTokens.set(token, {
    createdAt: Date.now(),
    expiresAt: Date.now() + (15 * 60 * 1000) // 15 minutes
  });

  // Attach token to response header
  res.setHeader('X-CSRF-Token', token);
  
  // Also attach to response locals for use in templates
  res.locals.csrfToken = token;
  
  next();
}

/**
 * Middleware to validate CSRF token
 * Only applies to state-changing operations (POST, PUT, DELETE, PATCH)
 */
function validateCSRFToken(req, res, next) {
  // Skip CSRF validation for:
  // - GET, HEAD, OPTIONS requests
  // - API endpoints using JWT (they have their own auth)
  // - File uploads
  if (['GET', 'HEAD', 'OPTIONS'].includes(req.method)) {
    return next();
  }

  if (req.path.startsWith('/api/')) {
    // API endpoints use JWT, CSRF less critical
    // But we can still validate if token is provided
    return next();
  }

  if (req.path.includes('/upload') || req.path.includes('/import')) {
    // File uploads handled separately
    return next();
  }

  // Get token from header or body
  const token = req.headers['x-csrf-token'] || req.body._csrf || req.query._csrf;

  if (!token) {
    return res.status(403).json({ 
      error: 'CSRF token missing',
      message: 'CSRF token is required for this operation'
    });
  }

  // Validate token
  const tokenData = csrfTokens.get(token);
  
  if (!tokenData) {
    return res.status(403).json({ 
      error: 'Invalid CSRF token',
      message: 'The CSRF token is invalid or expired'
    });
  }

  // Check expiration
  if (Date.now() > tokenData.expiresAt) {
    csrfTokens.delete(token);
    return res.status(403).json({ 
      error: 'CSRF token expired',
      message: 'The CSRF token has expired. Please refresh the page.'
    });
  }

  // Token is valid, remove it (one-time use) or keep for multiple uses
  // For now, we'll keep it until expiration
  next();
}

/**
 * Cleanup expired tokens (run periodically)
 */
function cleanupExpiredTokens() {
  const now = Date.now();
  for (const [token, data] of csrfTokens.entries()) {
    if (now > data.expiresAt) {
      csrfTokens.delete(token);
    }
  }
}

// Cleanup every 5 minutes
setInterval(cleanupExpiredTokens, 5 * 60 * 1000);

module.exports = {
  generateCSRFToken,
  validateCSRFToken,
  generateToken
};

