/**
 * Dashboard Controller
 * 
 * Handles dashboard metrics and statistics
 */

const BaseController = require('./BaseController');
const { asyncHandler } = require('../middleware/errorHandler');

class DashboardController extends BaseController {
  /**
   * Get dashboard metrics
   * GET /api/admin/dashboard
   */
  static getMetrics = asyncHandler(async (req, res) => {
    const connection = await BaseController.getConnection();

    try {
      // Today's messages
      const [todayMessages] = await connection.execute(`
        SELECT COUNT(*) as count 
        FROM messages 
        WHERE DATE(timestamp) = CURDATE()
      `);

      // Unique contacts today
      const [uniqueContacts] = await connection.execute(`
        SELECT COUNT(DISTINCT phone_number) as count 
        FROM messages 
        WHERE DATE(timestamp) = CURDATE()
      `);

      // Peak hours (last 7 days)
      const [hourlyStats] = await connection.execute(`
        SELECT 
          HOUR(timestamp) as hour,
          COUNT(*) as message_count
        FROM messages 
        WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 7 DAY)
        GROUP BY HOUR(timestamp)
        ORDER BY hour
      `);

      // Conversations by status
      const [conversationStats] = await connection.execute(`
        SELECT 
          COALESCE(status, 'active') as status,
          COUNT(*) as count
        FROM conversations
        GROUP BY status
      `);

      // Total messages per day (last 7 days)
      const [dailyMessages] = await connection.execute(`
        SELECT 
          DATE(timestamp) as date,
          COUNT(*) as count
        FROM messages
        WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 7 DAY)
        GROUP BY DATE(timestamp)
        ORDER BY date
      `);

      return BaseController.sendSuccess(res, {
        todayMessages: todayMessages[0]?.count || 0,
        uniqueContacts: uniqueContacts[0]?.count || 0,
        hourlyStats: hourlyStats || [],
        conversationStats: conversationStats || [],
        dailyMessages: dailyMessages || []
      });
    } finally {
      connection.release();
    }
  });
}

module.exports = DashboardController;

