/**
 * Base Controller
 * 
 * Provides common functionality for all controllers.
 * All controllers should extend this class to inherit common methods.
 * 
 * @class BaseController
 */
const { pool } = require('../config/database');
const { asyncHandler, DatabaseError } = require('../middleware/errorHandler');

class BaseController {
  /**
   * Get a database connection from the pool
   * 
   * @static
   * @async
   * @returns {Promise<Object>} Database connection object
   * @throws {DatabaseError} If connection cannot be established
   */
  static async getConnection() {
    try {
      return await pool.getConnection();
    } catch (error) {
      throw new DatabaseError('Failed to get database connection');
    }
  }

  /**
   * Execute a database query with automatic error handling
   * 
   * @static
   * @async
   * @param {string} query - SQL query string with placeholders (?)
   * @param {Array} [params=[]] - Array of parameters for the query
   * @param {Object} [connection=null] - Existing database connection (optional)
   * @returns {Promise<Array>} Query result array
   * @throws {DatabaseError} If query execution fails
   * @example
   * const users = await BaseController.executeQuery(
   *   'SELECT * FROM users WHERE id = ?',
   *   [1]
   * );
   */
  static async executeQuery(query, params = [], connection = null) {
    const shouldRelease = !connection;
    
    try {
      if (!connection) {
        connection = await this.getConnection();
      }

      const [result] = await connection.execute(query, params);
      return result;
    } catch (error) {
      throw new DatabaseError(`Database query failed: ${error.message}`);
    } finally {
      if (shouldRelease && connection) {
        connection.release();
      }
    }
  }

  /**
   * Send a standardized success response
   * 
   * @static
   * @param {Object} res - Express response object
   * @param {*} [data=null] - Response data to send
   * @param {number} [statusCode=200] - HTTP status code
   * @param {string} [message=null] - Optional success message
   * @returns {Object} Express response object
   * @example
   * return BaseController.sendSuccess(res, { user: userData }, 201, 'User created');
   */
  static sendSuccess(res, data = null, statusCode = 200, message = null) {
    const response = {
      success: true,
      ...(message && { message }),
      ...(data !== null && { data })
    };
    return res.status(statusCode).json(response);
  }

  /**
   * Send error response
   * @param {Object} res - Express response object
   * @param {string} message - Error message
   * @param {number} statusCode - HTTP status code
   */
  static sendError(res, message, statusCode = 400) {
    return res.status(statusCode).json({
      success: false,
      error: message
    });
  }

  /**
   * Paginate results
   * @param {Array} data - Data array
   * @param {number} page - Page number
   * @param {number} limit - Items per page
   * @returns {Object} Paginated result
   */
  static paginate(data, page = 1, limit = 10) {
    const offset = (page - 1) * limit;
    const total = data.length;
    const totalPages = Math.ceil(total / limit);
    const paginatedData = data.slice(offset, offset + limit);

    return {
      data: paginatedData,
      pagination: {
        page: parseInt(page),
        limit: parseInt(limit),
        total,
        totalPages
      }
    };
  }

  /**
   * Validate pagination parameters
   * @param {number} page - Page number
   * @param {number} limit - Items per page
   * @returns {Object} Validated pagination params
   */
  static validatePagination(page, limit) {
    const pageNum = Math.max(1, parseInt(page) || 1);
    const limitNum = Math.min(100, Math.max(1, parseInt(limit) || 10));
    
    return {
      page: pageNum,
      limit: limitNum,
      offset: (pageNum - 1) * limitNum
    };
  }
}

module.exports = BaseController;

