/**
 * Authentication Controller
 * 
 * Handles all authentication-related operations including login, logout, and token verification.
 * 
 * @class AuthController
 * @extends BaseController
 */
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const { body } = require('express-validator');
const BaseController = require('./BaseController');
const { asyncHandler, AuthenticationError, ValidationError } = require('../middleware/errorHandler');
const { validationRules, validateRequest } = require('../middleware/security');

class AuthController extends BaseController {
  /**
   * User login endpoint
   * 
   * Authenticates a user with username and password, returns JWT token on success.
   * 
   * @static
   * @route POST /api/auth/login
   * @param {Object} req - Express request object
   * @param {Object} req.body - Request body
   * @param {string} req.body.username - Username
   * @param {string} req.body.password - Password
   * @param {Object} res - Express response object
   * @returns {Promise<Object>} JSON response with token and user data
   * @throws {AuthenticationError} If credentials are invalid
   * @throws {ValidationError} If input validation fails
   */
  static login = [
    validationRules.username,
    validationRules.password,
    validateRequest,
    asyncHandler(async (req, res) => {
      const { username, password } = req.body;
      const connection = await BaseController.getConnection();

      try {
        // Find user
        const userQuery = `
          SELECT id, username, password, role, store AS store_name
          FROM users
          WHERE username = ?
        `;
        const [users] = await connection.execute(userQuery, [username]);

        if (users.length === 0) {
          throw new AuthenticationError('Invalid credentials');
        }

        const user = users[0];

        // Verify password
        const isPasswordValid = await bcrypt.compare(password, user.password);
        if (!isPasswordValid) {
          throw new AuthenticationError('Invalid credentials');
        }

        // Generate JWT token
        const tokenPayload = {
          userId: user.id,
          username: user.username,
          role: user.role,
          store: user.store_name
        };

        const token = jwt.sign(
          tokenPayload,
          process.env.JWT_SECRET,
          { expiresIn: '24h' }
        );

        return BaseController.sendSuccess(res, {
          token,
          user: {
            id: user.id,
            username: user.username,
            role: user.role,
            store: user.store_name
          }
        }, 200, 'Login successful');
      } finally {
        connection.release();
      }
    })
  ];

  /**
   * Verify JWT token validity
   * 
   * Checks if the provided JWT token is valid and not expired.
   * 
   * @static
   * @async
   * @route GET /api/auth/verify
   * @param {Object} req - Express request object
   * @param {Object} req.headers - Request headers
   * @param {string} req.headers.authorization - Bearer token
   * @param {Object} res - Express response object
   * @returns {Promise<Object>} JSON response with token validity and user data
   * @throws {AuthenticationError} If token is invalid or expired
   */
  static verify = asyncHandler(async (req, res) => {
    const authHeader = req.headers.authorization;
    const token = authHeader && authHeader.split(' ')[1];

    if (!token) {
      throw new AuthenticationError('Token not provided');
    }

    try {
      const decoded = jwt.verify(token, process.env.JWT_SECRET);
      return BaseController.sendSuccess(res, {
        valid: true,
        user: decoded
      });
    } catch (error) {
      throw new AuthenticationError('Invalid or expired token');
    }
  });

  /**
   * User logout endpoint
   * 
   * With JWT authentication, logout is handled client-side by removing the token.
   * This endpoint exists for consistency and can be used for logging purposes.
   * 
   * @static
   * @async
   * @route POST /api/auth/logout
   * @param {Object} req - Express request object
   * @param {Object} res - Express response object
   * @returns {Promise<Object>} JSON response confirming logout
   */
  static logout = asyncHandler(async (req, res) => {
    // With JWT, logout is handled client-side by removing the token
    // This endpoint is for consistency
    return BaseController.sendSuccess(res, null, 200, 'Logout successful');
  });
}

module.exports = AuthController;

