const { pool } = require('./database');

// English translations (default language)
const defaultTranslations = {
  // Menu
  "menu.dashboard": "Dashboard",
  "menu.conversations": "Conversations",
  "menu.stores": "Stores",
  "menu.departments": "Departments",
  "menu.users": "Users",
  "menu.contacts": "Contacts",
  "menu.faqs": "FAQs",
  "menu.payments": "Payments",
  "menu.ai_config": "AI Configuration",
  "menu.translations": "Translations",
  "menu.profile": "Edit Profile",
  "menu.edit_profile": "Edit Profile",

  // General buttons
  "btn.logout": "Logout",
  "btn.save": "Save",
  "btn.cancel": "Cancel",
  "btn.edit": "Edit",
  "btn.delete": "Delete",
  "btn.add": "Add",
  "btn.close": "Close",
  "btn.confirm": "Confirm",
  "btn.back": "Back",
  "btn.next": "Next",
  "btn.previous": "Previous",
  "btn.finish": "Finish",
  "btn.search": "Search",
  "btn.filter": "Filter",
  "btn.export": "Export",
  "btn.import": "Import",
  "btn.refresh": "Refresh",
  "btn.yes": "Yes",
  "btn.no": "No",
  "btn.select_time": "🕒 Select",

  // Dashboard
  "dashboard.title": "Dashboard",
  "dashboard.learn_tour": "Learn - Guided Tour",
  "dashboard.toggle_bot": "Toggle Bot",
  "dashboard.toggle_groups": "Toggle Groups",
  "dashboard.pause_bot": "Pause Bot",
  "dashboard.pause_groups": "Pause Groups",
  "dashboard.activate_bot": "Activate Bot",
  "dashboard.activate_groups": "Activate Groups",
  "dashboard.refresh_qr": "Refresh QR",
  "dashboard.disconnect_whatsapp": "Disconnect WhatsApp",
  "dashboard.messages_today": "Messages Today",
  "dashboard.messages": "Messages",
  "dashboard.unique_contacts": "Unique Contacts",
  "dashboard.waiting_conversations": "Waiting for Service",
  "dashboard.active_conversations": "In Service",
  "dashboard.messages_per_hour": "Messages per Hour",
  "dashboard.whatsapp_status": "WhatsApp Status",
  "dashboard.checking": "Checking...",
  "dashboard.loading_connection": "Loading connection status...",
  "dashboard.whatsapp_connected": "✅ WhatsApp Connected",
  "dashboard.whatsapp_disconnected": "⚠️ WhatsApp Disconnected",
  "dashboard.bot_ready": "Your bot is ready to use",
  "dashboard.bot_operational": "Bot is operational and ready to receive messages",
  "dashboard.connected": "Connected",

  // Users
  "users.title": "Manage Users",
  "users.add": "Add User",
  "users.edit": "Edit User",
  "users.search": "Search users...",
  "users.all_roles": "All roles",
  "users.role_admin": "Admin",
  "users.role_operator": "Operator",
  "users.role_viewer": "Viewer",
  "users.name": "Name",
  "users.role": "Role",
  "users.store": "Store",
  "users.department": "Department",
  "users.created_at": "Created at",
  "users.actions": "Actions",
  "users.active": "Active",

  // Conversations
  "conversations.title": "Conversations",
  "conversations.all_status": "All statuses",
  "conversations.status_waiting": "Waiting",
  "conversations.status_attended": "In service",
  "conversations.status_closed": "Closed",
  "conversations.all_stores": "All stores",
  "conversations.count": "conversations",
  "conversations.select": "Select a conversation",
  "conversations.select_message": "Select a conversation to view messages",
  "conversations.read_only": "Read-only mode - Admin cannot respond",
  "conversations.view_mode": "View mode - Read only",
  "conversations.attended_by": "Attended by:",

  // Stores
  "stores.title": "Manage Stores",
  "stores.add": "Add Store",
  "stores.edit": "Edit Store",
  "stores.name": "Name",
  "stores.description": "Description",
  "stores.active": "Active",
  "stores.users": "Users",
  "stores.created_at": "Created at",
  "stores.actions": "Actions",

  // Departments
  "departments.title": "Manage Departments",
  "departments.add": "Add Department",
  "departments.edit": "Edit Department",
  "departments.name": "Name",
  "departments.description": "Description",
  "departments.users": "Users",
  "departments.created_at": "Created at",
  "departments.actions": "Actions",
  "departments.active": "Active",

  // FAQs
  "faqs.title": "Manage FAQs",
  "faqs.add": "Add FAQ",
  "faqs.search": "Search FAQs...",
  "faqs.all_categories": "All categories",
  "faqs.category_general": "General",
  "faqs.category_products": "Products",
  "faqs.category_payments": "Payments",
  "faqs.category_delivery": "Delivery",
  "faqs.question": "Question",
  "faqs.questions": "Question(s)",
  "faqs.multiple_questions_hint": "You can add multiple ways to ask by separating with commas",
  "faqs.multiple_questions_example": "Example: What time?, What time do you open?, Opening hours?",
  "faqs.answer": "Answer",
  "faqs.emoji": "Emoji",
  "faqs.key": "Key",
  "faqs.actions": "Actions",
  "faqs.welcome_title": "Welcome Messages",
  "faqs.welcome_desc": "Configure messages to be sent when a customer starts a conversation (only 1x per day)",
  "faqs.welcome_msg1": "Message 1",
  "faqs.welcome_msg1_placeholder": "Type the first message...",
  "faqs.welcome_msg2": "Message 2",
  "faqs.welcome_msg2_placeholder": "Type the second message...",
  "faqs.welcome_msg3": "Message 3 (with optional media)",
  "faqs.welcome_msg3_placeholder": "Type the third message...",
  "faqs.welcome_placeholders": "Placeholders:",
  "faqs.welcome_media": "Media (optional):",
  "faqs.welcome_save": "Save Welcome Messages",
  "faqs.welcome_preview": "Mobile Preview",
  "faqs.welcome_preview_empty": "Messages will appear here as you type",
  "faqs.reaction_time": "Reaction Time (seconds)",
  "faqs.response_time": "Response Time (seconds)",
  "faqs.schedule_hours": "Business Hours",
  "faqs.schedule_days": "Days of the Week",
  "faqs.order_position": "Display Order",
  "faqs.test_faq": "Test FAQ",
  "faqs.no_messages": "No messages configured",
  "faqs.message_saved": "Welcome messages saved successfully!",
  "faqs.message_error": "Error saving welcome messages",
  "faqs.faq_saved": "FAQ saved successfully!",
  "faqs.faq_error": "Error saving FAQ",
  "faqs.faq_deleted": "FAQ deleted successfully!",
  "faqs.faq_delete_error": "Error deleting FAQ",
  "faqs.confirm_delete": "Are you sure you want to delete this FAQ?",

  // Payments
  "payments.title": "Manage Payments",
  "payments.config_gateways": "Configure Gateways",
  "payments.today": "Payments Today",
  "payments.total_amount": "Total Amount",
  "payments.pending": "Pending",
  "payments.success_rate": "Success Rate",
  "payments.all_status": "All statuses",
  "payments.status_pending": "Pending",
  "payments.status_completed": "Completed",
  "payments.status_failed": "Failed",
  "payments.status_cancelled": "Cancelled",
  "payments.id": "ID",
  "payments.customer": "Customer",
  "payments.amount": "Amount",
  "payments.gateway": "Gateway",
  "payments.status": "Status",
  "payments.date": "Date",
  "payments.actions": "Actions",
  "payments.loading": "Loading payments...",

  // Translations
  "translations.title": "Manage Translations",
  "translations.import": "Import",
  "translations.export": "Export",
  "translations.add_language": "Add Language",
  "translations.apply_system": "Apply to System",
  "translations.active_language": "Active Language:",
  "translations.total_strings": "Total Strings",
  "translations.translated_strings": "Translated Strings",
  "translations.completion_rate": "Completion Rate",
  "translations.tab_languages": "Languages",
  "translations.tab_strings": "Strings",
  "translations.code": "Code",
  "translations.name": "Name",
  "translations.strings": "Strings",
  "translations.completeness": "Completeness",
  "translations.default": "Default",
  "translations.actions": "Actions",
  "translations.search": "Search strings...",
  "translations.all_categories": "All categories",
  "translations.add_string": "Add String",
  "translations.key": "Key",
  "translations.value": "Value",
  "translations.category": "Category",
  "translations.edit_string": "Edit String",
  "translations.language": "Language",
  "translations.language_code": "Language Code",
  "translations.language_name": "Language Name",
  "translations.copy_from_default": "Copy strings from default language",
  "translations.set_default": "Set as Default",

  // AI Configuration
  "ai_config.title": "AI Configuration",
  "ai_config.test_connection": "Test Connection",
  "ai_config.deepseek": "DeepSeek",
  "ai_config.openai": "OpenAI",
  "ai_config.api_key": "API Key:",
  "ai_config.api_key_placeholder": "Your API key",
  "ai_config.persona_name": "Persona Name:",
  "ai_config.persona_name_placeholder": "AI Assistant",
  "ai_config.persona_description": "Persona Description:",
  "ai_config.persona_description_placeholder": "I am an intelligent virtual assistant...",
  "ai_config.working_hours": "Business Hours:",
  "ai_config.working_hours_placeholder": "00:00-23:59",
  "ai_config.model": "Model:",
  "ai_config.temperature": "Temperature:",
  "ai_config.temperature_desc": "Controls response creativity (0.0 = more precise, 2.0 = more creative)",
  "ai_config.max_tokens": "Max Tokens:",
  "ai_config.max_tokens_desc": "Token limit for AI responses",
  "ai_config.save_deepseek": "Save DeepSeek",
  "ai_config.save_openai": "Save OpenAI",
  "ai_config.loading": "Loading AI settings...",

  // Time Selection Modal
  "time_picker.title": "Select Time",
  "time_picker.start_time": "Start Time:",
  "time_picker.end_time": "End Time:",

  // Contacts
  "contacts.title": "Manage Contacts",
  "contacts.add": "Add Contact",
  "contacts.tab_contacts": "Contacts",
  "contacts.tab_groups": "Groups",
  "contacts.search": "Search contacts...",
  "contacts.all_groups": "All groups",
  "contacts.import_csv": "Import CSV",
  "contacts.name": "Name",
  "contacts.phone": "Phone",
  "contacts.email": "Email",
  "contacts.group": "Group",
  "contacts.tags": "Tags",
  "contacts.created_at": "Created at",
  "contacts.actions": "Actions",
  "contacts.no_contacts": "No contacts found",
  "contacts.edit": "Edit Contact",
  "contacts.delete": "Delete Contact",
  "contacts.notes": "Notes",
  "contacts.groups_title": "Manage Groups",
  "contacts.add_group": "Add Group",
  "contacts.edit_group": "Edit Group",
  "contacts.group_name": "Group Name",
  "contacts.group_description": "Group Description",
  "contacts.no_groups": "No groups found",
  "contacts.import_title": "Import Contacts",
  "contacts.select_file": "Select CSV file",
  "contacts.select_group": "Select group (optional)",
  "contacts.file_format": "File format:",
  "contacts.imported": "contacts imported",
  "contacts.duplicates": "duplicates ignored",
  "contacts.group_contacts": "Contacts",
  "contacts.loading": "Loading contacts...",
  "contacts.new_group": "New Group",

  // Conversations
  "conversations.back": "Back",

  // Logout
  "logout.confirm_message": "Are you sure you want to logout?",

  // Mobile
  "mobile.home": "Home",
  "mobile.menu": "Menu",
  "mobile.payments": "Payments",
  "mobile.users": "Users",
  "mobile.whatsapp": "WhatsApp",

  // Profile
  "profile.title": "Edit Profile",
  "profile.button_colors": "Button Colors",
  "profile.card_background": "Card Background",
  "profile.card_border": "Card Border",
  "profile.card_colors": "Card Colors",
  "profile.color_customization": "Color Customization",
  "profile.crop_image": "Crop Image",
  "profile.danger_button": "Danger Button",
  "profile.favicon": "Favicon",
  "profile.login_logo": "Login Logo",
  "profile.max_size": "Max size",
  "profile.menu_active": "Active Item",
  "profile.menu_background": "Menu Background",
  "profile.menu_colors": "Menu Colors",
  "profile.menu_hover": "Menu Hover",
  "profile.preview": "Preview",
  "profile.preview_login": "Preview Login Screen",
  "profile.primary_button": "Primary Button",
  "profile.primary_color": "Primary Color",
  "profile.remove_logo": "Remove Logo",
  "profile.reset_colors": "Reset Default Colors",
  "profile.success_button": "Success Button",
  "profile.system_logo": "System Logo",
  "profile.system_logos": "System Logos",
  "profile.system_name": "System Name",
  "profile.system_name_placeholder": "My Company",
  "profile.text_color": "Text Color",
  "profile.text_colors": "Text Colors",
  "profile.title_color": "Title Color",
  "profile.upload_favicon": "Upload Favicon",
  "profile.upload_logo": "Upload Logo",
  "profile.warning_button": "Warning Button",

  // User
  "user.attention": "Attention",
  "user.audio": "Audio",
  "user.camera": "Camera",
  "user.confirm_transfer": "Confirm Transfer",
  "user.department": "Department",
  "user.document": "Document",
  "user.gallery": "Gallery",
  "user.location": "Location",
  "user.logout_message": "Are you sure you want to logout?",
  "user.logout_title": "Logout",
  "user.payment": "Payment",
  "user.select_department_message": "Select a department",
  "user.select_option": "Select an option",
  "user.select_store_message": "Select a store",
  "user.store": "Store",
  "user.transfer_store_warning": "Transfer to store?",
  "user.transfer_support_warning": "Transfer to support?",
  "user.transfer_to_store": "Transfer to Store",
  "user.transfer_to_support": "Transfer to Support",

  // Modal strings
  "action_add.add_contact": "Add Contact",
  "label.file_too_large_maximum_16mb": "File too large. Maximum: 16MB",
  "field_label.hours": "Hours:",
  "general.import": "Import",
  "contacts.import_contacts": "Import Contacts",
  "translations.import_translations": "Import Translations",
  "whatsapp.starting_whatsapp_reconnection": "Starting WhatsApp reconnection...",
  "success.loja_atualizada_com_sucesso": "Store updated successfully",
  "stores.store": "Store:",
  "field_label.message": "Message:",
  "success.mensagens_de_boas-vindas_salvas_com": "Welcome messages saved successfully!",
  "contacts.nenhum_contato_encontrado": "No contacts found",
  "success.campaign_created_successfully": "Campaign created successfully",
  "general.nenhuma_mensagem_nesta_conversa": "No messages in this conversation",
  "campaigns.campaign_name": "Campaign Name:",
  "stores.store_name": "Store Name:",
  "campaigns.nome_da_campanha_e_mensagem": "Campaign name and message are required",
  "users.username": "Username:",
  "groups.group_name": "Group Name:",
  "departments.department_name": "Department Name:",
  "groups.group_name_is_required": "Group name is required",
  "departments.department_name_is_required": "Department name is required",
  "success.nome_do_sistema_atualizado_com": "System name updated successfully!",
  "general.cancel": "Cancel",
  "field_label.name": "Name:",
  "success.notification_created_successfully": "Notification created successfully",
  "action_add.new_campaign": "New Campaign",
  "action_add.new_push_notification": "New Push Notification",
  "action_add.new_password_leave_blank_to": "New Password (leave blank to keep current):",
  "general.nova_senha_opcional": "New password (optional)",
  "action_add.new_contact_group": "New Contact Group",
  "campaigns.o_sistema_de_campanhas_será": "Campaign system will be implemented soon.",
  "contacts.por_enquanto_você_pode_gerenciar": "For now, you can manage contact groups in the \"Contact Groups\" tab.",
  "general.por_favor_insira_um_nome": "Please enter a name for the system",
  "loading.loading_push_notifications": "Loading push notifications...",
  "general.please_select_a_reaction_emoji": "Please select a reaction emoji",
  "payments.preencha_client_id_e_client": "Fill in PayPal Client ID and Client Secret",
  "payments.preencha_email_e_token_do": "Fill in PagSeguro email and token",
  "general.fill_in_all_required_fields": "Fill in all required fields",
  "general.promotions": "Promotions",
  "field_label.promotions": "Promotions:",
  "action_add.reconnection_started_new_qr_code": "Reconnection started. New QR Code will be generated shortly.",
  "action_save.save": "Save",
  "action_save.save_changes": "Save Changes",
  "action_save.save_pagseguro": "Save PagSeguro",
  "success.settings_saved_successfully": "Settings saved successfully",
  "action_save.save_paypal": "Save PayPal",
  "contacts.selecione_pelo_menos_um_contato": "Select at least one contact",
  "general.select_a_csv_file": "Select a CSV file",
  "general.selecione_um_emoji": "Select an emoji",
  "groups.selecione_um_grupo": "Select a group",
  "contacts.selecione_um_grupo_de_contatos": "Select a contact group",
  "general.selecione_um_idioma_e_um": "Select a language and a file",
  "departments.selecione_um_setor": "Select a department",
  "field_label.senha": "Password:",
  "departments.department": "Department:",
  "success.contact_added_successfully": "Contact added successfully",
  "campaigns.campaign_system_not_yet_available": "Campaign system not yet available",
  "campaigns.campaign_system_not_yet_available_1": "Campaign system not yet available in backend",
  "translations.translation_system_not_initialized": "Translation system not initialized",
  "field_label.tags_comma_separated": "Tags (comma separated):",
  "field_label.phone": "Phone:",
  "general.testing_ai_connection": "Testing AI connection...",
  "general.token_de_integração": "Integration token",
  "field_label.title": "Title:",
  "field_label.url_optional": "URL (optional):",
  "general.url_do_ícone": "Icon URL",
  "success.contact_updated_successfully": "Contact updated successfully",
  "campaigns.campaign_view_will_be_implemented": "Campaign view will be implemented soon",
  "success.whatsapp_desconectado_com_sucesso": "WhatsApp disconnected successfully",
  "general.seu-emailexemplocom": "your-email@example.com",
  "field_label.icon_optional": "Icon (optional):",
  "contacts.este_grupo_não_possui_contatos": "⚠️ This group has no contacts.",
  "contacts.contactslength_contatos_selecionados": "✅ ${contacts.length} contact(s) selected",
  "contacts.selectedmasscontactslength_contatos_selecionados": "✅ ${selectedMassContacts.length} contact(s) selected",
  "general.mídia_anexada": "📎 Media attached",
  "general.refresh_qr_code": "🔄 Refresh QR Code",
  "success.contact_deleted_successfully": "Contact deleted successfully",
  "stores.colors_restored_to_default": "Colors restored to default!",
  "success.colors_saved_and_applied_successfully": "Colors saved and applied successfully!",
  "action_add.add_faq": "Add FAQ",
  "action_add.create_campaign": "Create Campaign",
  "action_add.create_group": "Create Group",
  "departments.optional_department_description": "Optional department description",
  "field_label.description": "Description:",
  "general.type_at_least_2_characters": "Type at least 2 characters to search",
  "general.type_your_message_here": "Type your message here...",
  "general.type_a_message_or_attach": "Type a message or attach media",
  "action_edit.edit": "Edit",
  "action_edit.edit_contact": "Edit Contact",
  "action_edit.edit_faq": "Edit FAQ",
  "action_add.add_store": "Add Store",
  "action_edit.edit_group": "Edit Group",
  "action_edit.campaign_editing_will_be_implemented": "Campaign editing will be implemented soon",
  "field_label.email_optional": "Email (optional):",
  "payments.pagseguro_email": "PagSeguro Email:",
  "field_label.reaction_emoji": "Reaction Emoji:",
  "action_add.address": "Address",
  "action_add.address_1": "Address:",
  "notifications.send_notification": "Send Notification",
  "error.error_adding_contact": "Error adding contact",
  "error.error_updating_faq": "Error updating FAQ",
  "action_add.add_department": "Add Department",
  "error.error_updating_contact": "Error updating contact",
  "error.error_updating_group": "Error updating group",
  "error.error_updating_store": "Error updating store",
  "error.error_fetching_contacts": "Error fetching contacts",
  "error.error_loading_faqs": "Error loading FAQs",
  "error.error_loading_ai_configuration": "Error loading AI configuration",
  "error.error_loading_settings": "Error loading settings",
  "error.error_loading_ai_settings": "Error loading AI settings",
  "error.error_loading_contact": "Error loading contact",
  "error.error_loading_contacts": "Error loading contacts",
  "action_add.add_user": "Add User",
  "error.error_loading_conversations": "Error loading conversations",
  "error.error_loading_group": "Error loading group",
  "error.error_loading_messages": "Error loading messages",
  "error.error_loading_push_notifications": "Error loading push notifications",
  "error.error_loading_payments": "Error loading payments",
  "error.erro_ao_criar_faq": "Error creating FAQ",
  "error.error_creating_notification": "Error creating notification",
  "error.error_creating_user": "Error creating user",
  "error.erro_ao_desconectar_whatsapp": "Error disconnecting WhatsApp",
  "error.error_deleting_faq": "Error deleting FAQ",
  "general.admin_cannot_send_messages_read-only": "Admin cannot send messages. Read-only mode.",
  "error.error_deleting_contact": "Error deleting contact",
  "error.error_deleting_group": "Error deleting group",
  "error.error_deleting_store": "Error deleting store",
  "error.erro_ao_excluir_setor": "Error deleting department",
  "error.erro_ao_filtrar_conversas": "Error filtering conversations",
  "error.erro_ao_importar_contatos": "Error importing contacts",
  "error.error_importing_translations": "Error importing translations",
  "error.erro_ao_recarregar_grupos": "Error reloading groups",
  "error.error_saving_faq": "Error saving FAQ",
  "error.error_saving_settings": "Error saving settings",
  "field_label.schedule_for_optional": "Schedule for (optional):",
  "error.erro_ao_salvar_mensagens_de": "Error saving welcome messages",
  "error.error_testing_ai_connection": "Error testing AI connection",
  "error.connection_error_creating_campaign": "Connection error creating campaign",
  "error.connection_error_creating_group": "Connection error creating group",
  "error.connection_error_reconnecting_whatsapp": "Connection error reconnecting WhatsApp",
  "error.form_error": "Form error",
  "error.connection_test_error": "Connection test error",
  "error.error_modal_container_not_found": "Error: Modal container not found",
  "error.error_faq_not_loaded": "Error: FAQ not loaded",
  "general.esta_funcionalidade_será_implementada_em": "This feature will be implemented soon.",
  "field_label.csv_file": "CSV File:",
  "label.ex_promoção_de_verão": "Ex: Promoção de Verão",
  "action_delete.excluir": "Delete",
  "campaigns.campaign_deletion_will_be_implemented": "Campaign deletion will be implemented soon",
  "success.faq_added_successfully": "FAQ added successfully",
  "success.faq_updated_successfully": "FAQ updated successfully",
  "success.faq_deleted_successfully": "FAQ deleted successfully",
  "success.faq_deleted_successfully_1": "FAQ deleted successfully",
  "translations.translation_filter_applied": "Translation filter applied",
  "groups.group_filter_will_be_implemented": "Group filter will be implemented soon",
  "label.expected_format_namephoneemail_header_in": "Expected format: name,phone,email (header in first line)",
  "field_label.json_file": "JSON File:",
  "action_edit.faq_editing_function_not_implemented": "FAQ editing function not implemented",
  "general.preview_function_in_development": "Preview function in development",
  "success.grupo_atualizado_com_sucesso": "Group updated successfully",
  "success.group_created_successfully": "Group created successfully",
  "contacts.contact_group": "Contact Group:",
  "groups.target_group_optional": "Target Group (optional):",
  "success.group_deleted_successfully": "Group deleted successfully",
  "groups.grupo": "Group:",
  "success.grupos_recarregados_com_sucesso": "Groups reloaded successfully",
  "general.hours": "Hours",

  // Notifications
  "notification.language_changed": "Language changed successfully",
  "notification.confirm_disconnect_whatsapp": "Are you sure you want to disconnect WhatsApp?",
  "notification.confirm_reset_session": "This will clear all session data and require new authentication. Continue?",
  "notification.confirm_disconnect_whatsapp_full": "Are you sure you want to disconnect WhatsApp? This will stop all automatic responses.",
  "notification.confirm_delete_faq": "Are you sure you want to delete this FAQ?",
  "notification.confirm_delete_contact": "Are you sure you want to delete this contact?",
  "notification.confirm_delete_group": "Are you sure you want to delete this group? Contacts will not be deleted, only unlinked from the group.",
  "notification.confirm_delete_department": "Are you sure you want to delete this department? This action cannot be undone.",
  "notification.confirm_delete_campaign": "Are you sure you want to delete this campaign?",
  "notification.confirm_delete_store": "Are you sure you want to delete this store?",
  "notification.faq_create_error": "Error creating FAQ",
  "notification.whatsapp_reconnect_error": "Connection error reconnecting WhatsApp",
  "notification.whatsapp_disconnect_error": "Error disconnecting WhatsApp",
  "notification.test_connection_error": "Connection test error",
  "notification.conversations_load_error": "Error loading conversations",
  "notification.messages_load_error": "Error loading messages",
  "notification.store_update_error": "Error updating store",
  "notification.faqs_load_error": "Error loading FAQs",
  "notification.config_save_error": "Error saving settings",
  "notification.config_load_error": "Error loading settings",
  "notification.modal_container_error": "Error: Modal container not found",
  "notification.select_department": "Select a department",
  "notification.faq_delete_error": "Error deleting FAQ",
  "notification.mass_send_load_error": "Error loading mass send: ",
  "notification.payments_load_error": "Error loading payments",
  "notification.translations_not_initialized": "Translation system not initialized",
  "notification.translations_load_error": "Error loading translations: ",
  "notification.ai_config_load_error": "Error loading AI configuration",
  "notification.push_notifications_load_error": "Error loading push notifications",
  "notification.select_emoji": "Please select a reaction emoji",
  "notification.faq_save_error": "Error saving FAQ",
  "notification.fill_required_fields": "Fill in all required fields",
  "notification.campaign_create_error": "Connection error creating campaign",
  "notification.select_csv_file": "Select a CSV file",
  "notification.contacts_import_error": "Error importing contacts",
  "notification.group_name_required": "Group name is required",
  "notification.group_create_error": "Connection error creating group",
  "notification.form_error": "Form error",
  "notification.pagseguro_fill_fields": "Fill in PagSeguro email and token",
  "notification.paypal_fill_fields": "Fill in PayPal Client ID and Client Secret",
  "notification.select_language_and_file": "Select a language and a file",
  "notification.invalid_time_values": "Invalid time values. Please check the hours and minutes.",
  "notification.select_category": "Please select a category",
  "notification.key_required": "Translation key is required",
  "notification.value_required": "Translation value is required",
  "notification.invalid_key_format": "Key must contain only letters, numbers and underscores",
  "notification.string_added_success": "Translation string added successfully to all languages",
  "notification.string_add_error": "Error adding translation string",
  "notification.translations_import_error": "Error importing translations",
  "notification.user_create_error": "Error creating user",
  "notification.store_delete_error": "Error deleting store",
  "notification.conversations_filter_error": "Error filtering conversations",
  "notification.click_field_before_placeholder": "Click on a text field before using the placeholder",
  "notification.notification_create_error": "Error creating notification",
  "notification.configure_deepseek_first": "Configure DeepSeek API Key first",
  "notification.ai_test_connection_error": "Error testing AI connection",
  "notification.faq_edit_load_error": "Error loading FAQ for editing: ",
  "notification.faq_not_loaded": "Error: FAQ not loaded",
  "notification.faq_update_error": "Error updating FAQ",
  "notification.contacts_load_error": "Error loading contacts",
  "notification.contact_add_error": "Error adding contact",
  "notification.contact_load_error": "Error loading contact",
  "notification.contact_update_error": "Error updating contact",
  "notification.contact_delete_error": "Error deleting contact",
  "notification.group_load_error": "Error loading group",
  "notification.group_update_error": "Error updating group",
  "notification.group_delete_error": "Error deleting group",
  "notification.campaigns_load_error": "Error loading campaigns: ",
  "notification.groups_reload_error": "Error reloading groups",
  "notification.campaigns_not_available": "Campaign system not yet available",
  "notification.campaign_name_message_required": "Campaign name and message are required",
  "notification.select_contact_group": "Select a contact group",
  "notification.campaigns_backend_not_available": "Campaign system not yet available in backend",
  "notification.department_name_required": "Department name is required",
  "notification.department_delete_error": "Error deleting department",
  "notification.file_too_large": "File too large. Maximum: 16MB",
  "notification.message_or_media_required": "Type a message or attach media",
  "notification.select_at_least_one_contact": "Select at least one contact",
  "notification.select_group": "Select a group",
  "notification.enter_system_name": "Please enter a system name",
  "notification.welcome_messages_save_error": "Error saving welcome messages",
  "notification.reconnection_started": "Reconnection started. New QR Code will be generated shortly.",
  "notification.whatsapp_disconnected": "WhatsApp disconnected successfully",
  "notification.store_updated": "Store updated successfully",
  "notification.config_saved": "Settings saved successfully",
  "notification.faq_deleted": "FAQ deleted successfully",
  "notification.faq_added": "FAQ added successfully",
  "notification.campaign_created": "Campaign created successfully",
  "notification.group_created": "Group created successfully",
  "notification.notification_created": "Notification created successfully",
  "notification.faq_updated": "FAQ updated successfully",
  "notification.contact_added": "Contact added successfully",
  "notification.contact_updated": "Contact updated successfully",
  "notification.contact_deleted": "Contact deleted successfully",
  "notification.group_updated": "Group updated successfully",
  "notification.group_deleted": "Group deleted successfully",
  "notification.groups_reloaded": "Groups reloaded successfully",
  "notification.system_name_updated": "System name updated successfully!",
  "notification.colors_saved": "Colors saved and applied successfully!",
  "notification.colors_restored": "Colors restored to default!",
  "notification.placeholder_copied": "Placeholder copied: ",
  "notification.welcome_messages_saved": "Welcome messages saved successfully!",
  "notification.starting_reconnection": "Starting WhatsApp reconnection...",
  "notification.disconnecting_whatsapp": "Disconnecting WhatsApp...",
  "notification.loading_push_notifications": "Loading push notifications...",
  "notification.payments_filter_applied": "Payments filter applied",
  "notification.translations_filter_applied": "Translations filter applied",
  "notification.group_filter_coming_soon": "Group filter will be implemented soon",
  "notification.testing_ai_connection": "Testing AI connection...",
  "notification.preview_in_development": "Preview function in development",
  "notification.admin_readonly": "Admin cannot send messages. Read-only mode.",
  "notification.edit_faq_not_implemented": "FAQ editing function not implemented",
  "notification.view_campaign_coming_soon": "Campaign view will be implemented soon",
  "notification.edit_campaign_coming_soon": "Campaign editing will be implemented soon",
  "notification.delete_campaign_coming_soon": "Campaign deletion will be implemented soon",

  // Modal strings
  "action_add.add_contact": "Add Contact",
  "label.file_too_large_maximum_16mb": "File too large. Maximum: 16MB",
  "field_label.hours": "Hours:",
  "general.import": "Import",
  "contacts.import_contacts": "Import Contacts",
  "translations.import_translations": "Import Translations",
  "whatsapp.starting_whatsapp_reconnection": "Starting WhatsApp reconnection...",
  "success.loja_atualizada_com_sucesso": "Store updated successfully",
  "stores.store": "Store:",
  "field_label.message": "Message:",
  "success.mensagens_de_boas-vindas_salvas_com": "Welcome messages saved successfully!",
  "contacts.nenhum_contato_encontrado": "No contacts found",
  "success.campaign_created_successfully": "Campaign created successfully",
  "general.nenhuma_mensagem_nesta_conversa": "No messages in this conversation",
  "campaigns.campaign_name": "Campaign Name:",
  "stores.store_name": "Store Name:",
  "campaigns.nome_da_campanha_e_mensagem": "Campaign name and message are required",
  "users.username": "Username:",
  "groups.group_name": "Group Name:",
  "departments.department_name": "Department Name:",
  "groups.group_name_is_required": "Group name is required",
  "departments.department_name_is_required": "Department name is required",
  "success.nome_do_sistema_atualizado_com": "System name updated successfully!",
  "general.cancel": "Cancel",
  "field_label.name": "Name:",
  "success.notification_created_successfully": "Notification created successfully",
  "action_add.new_campaign": "New Campaign",
  "action_add.new_push_notification": "New Push Notification",
  "action_add.new_password_leave_blank_to": "New Password (leave blank to keep current):",
  "general.nova_senha_opcional": "New password (optional)",
  "action_add.new_contact_group": "New Contact Group",
  "campaigns.o_sistema_de_campanhas_será": "Campaign system will be implemented soon.",
  "general.por_favor_insira_um_nome": "Please enter a name for the system",
  "loading.loading_push_notifications": "Loading push notifications...",
  "general.please_select_a_reaction_emoji": "Please select a reaction emoji",
  "payments.preencha_client_id_e_client": "Fill in PayPal Client ID and Client Secret",
  "payments.preencha_email_e_token_do": "Fill in PagSeguro email and token",
  "general.fill_in_all_required_fields": "Fill in all required fields",
  "general.promotions": "Promotions",
  "field_label.promotions": "Promotions:",
  "action_add.reconnection_started_new_qr_code": "Reconnection started. New QR Code will be generated shortly.",
  "action_save.save": "Save",
  "action_save.save_changes": "Save Changes",
  "action_save.save_pagseguro": "Save PagSeguro",
  "success.settings_saved_successfully": "Settings saved successfully",
  "action_save.save_paypal": "Save PayPal",
  "contacts.selecione_pelo_menos_um_contato": "Select at least one contact",
  "general.select_a_csv_file": "Select a CSV file",
  "general.selecione_um_emoji": "Select an emoji",
  "groups.selecione_um_grupo": "Select a group",
  "contacts.selecione_um_grupo_de_contatos": "Select a contact group",
  "general.selecione_um_idioma_e_um": "Select a language and a file",
  "departments.selecione_um_setor": "Select a department",
  "field_label.senha": "Password:",
  "departments.department": "Department:",
  "success.contact_added_successfully": "Contact added successfully",
  "campaigns.campaign_system_not_yet_available": "Campaign system not yet available",
  "campaigns.campaign_system_not_yet_available_1": "Campaign system not yet available in backend",
  "translations.translation_system_not_initialized": "Translation system not initialized",
  "field_label.tags_comma_separated": "Tags (comma separated):",
  "field_label.phone": "Phone:",
  "general.testing_ai_connection": "Testing AI connection...",
  "general.token_de_integração": "Integration token",
  "field_label.title": "Title:",
  "field_label.url_optional": "URL (optional):",
  "general.url_do_ícone": "Icon URL",
  "success.contact_updated_successfully": "Contact updated successfully",
  "campaigns.campaign_view_will_be_implemented": "Campaign view will be implemented soon",
  "success.whatsapp_desconectado_com_sucesso": "WhatsApp disconnected successfully",
  "general.seu-emailexemplocom": "your-email@example.com",
  "field_label.icon_optional": "Icon (optional):",
  "contacts.este_grupo_não_possui_contatos": "⚠️ This group has no contacts.",
  "contacts.contactslength_contatos_selecionados": "✅ ${contacts.length} contact(s) selected",
  "contacts.selectedmasscontactslength_contatos_selecionados": "✅ ${selectedMassContacts.length} contact(s) selected",
  "general.mídia_anexada": "📎 Media attached",
  "general.refresh_qr_code": "🔄 Refresh QR Code",
  "success.contact_deleted_successfully": "Contact deleted successfully",
  "stores.colors_restored_to_default": "Colors restored to default!",
  "success.colors_saved_and_applied_successfully": "Colors saved and applied successfully!",
  "action_add.add_faq": "Add FAQ",
  "action_add.create_campaign": "Create Campaign",
  "action_add.create_group": "Create Group",
  "departments.optional_department_description": "Optional department description",
  "field_label.description": "Description:",
  "general.type_at_least_2_characters": "Type at least 2 characters to search",
  "general.type_your_message_here": "Type your message here...",
  "general.type_a_message_or_attach": "Type a message or attach media",
  "action_edit.edit": "Edit",
  "action_edit.edit_contact": "Edit Contact",
  "action_edit.edit_faq": "Edit FAQ",
  "action_add.add_store": "Add Store",
  "action_edit.edit_group": "Edit Group",
  "action_edit.campaign_editing_will_be_implemented": "Campaign editing will be implemented soon",
  "field_label.email_optional": "Email (optional):",
  "payments.pagseguro_email": "PagSeguro Email:",
  "field_label.reaction_emoji": "Reaction Emoji:",
  "action_add.address": "Address",
  "action_add.address_1": "Address:",
  "notifications.send_notification": "Send Notification",
  "error.error_adding_contact": "Error adding contact",
  "error.error_updating_faq": "Error updating FAQ",
  "action_add.add_department": "Add Department",
  "error.error_updating_contact": "Error updating contact",
  "error.error_updating_group": "Error updating group",
  "error.error_updating_store": "Error updating store",
  "error.error_fetching_contacts": "Error fetching contacts",
  "error.error_loading_faqs": "Error loading FAQs",
  "error.error_loading_ai_configuration": "Error loading AI configuration",
  "error.error_loading_settings": "Error loading settings",
  "error.error_loading_ai_settings": "Error loading AI settings",
  "error.error_loading_contact": "Error loading contact",
  "error.error_loading_contacts": "Error loading contacts",
  "action_add.add_user": "Add User",
  "error.error_loading_conversations": "Error loading conversations",
  "error.error_loading_group": "Error loading group",
  "error.error_loading_messages": "Error loading messages",
  "error.error_loading_push_notifications": "Error loading push notifications",
  "error.error_loading_payments": "Error loading payments",
  "error.erro_ao_criar_faq": "Error creating FAQ",
  "error.error_creating_notification": "Error creating notification",
  "error.error_creating_user": "Error creating user",
  "error.erro_ao_desconectar_whatsapp": "Error disconnecting WhatsApp",
  "error.error_deleting_faq": "Error deleting FAQ",
  "general.admin_cannot_send_messages_read-only": "Admin cannot send messages. Read-only mode.",
  "error.error_deleting_contact": "Error deleting contact",
  "error.error_deleting_group": "Error deleting group",
  "error.error_deleting_store": "Error deleting store",
  "error.erro_ao_excluir_setor": "Error deleting department",
  "error.erro_ao_filtrar_conversas": "Error filtering conversations",
  "error.erro_ao_importar_contatos": "Error importing contacts",
  "error.error_importing_translations": "Error importing translations",
  "error.erro_ao_recarregar_grupos": "Error reloading groups",
  "error.error_saving_faq": "Error saving FAQ",
  "error.error_saving_settings": "Error saving settings",
  "field_label.schedule_for_optional": "Schedule for (optional):",
  "error.erro_ao_salvar_mensagens_de": "Error saving welcome messages",
  "error.error_testing_ai_connection": "Error testing AI connection",
  "error.connection_error_creating_campaign": "Connection error creating campaign",
  "error.connection_error_creating_group": "Connection error creating group",
  "error.connection_error_reconnecting_whatsapp": "Connection error reconnecting WhatsApp",
  "error.form_error": "Form error",
  "error.connection_test_error": "Connection test error",
  "error.error_modal_container_not_found": "Error: Modal container not found",
  "error.error_faq_not_loaded": "Error: FAQ not loaded",
  "general.esta_funcionalidade_será_implementada_em": "This feature will be implemented soon.",
  "field_label.csv_file": "CSV File:",
  "label.ex_promoção_de_verão": "Ex: Promoção de Verão",
  "action_delete.excluir": "Delete",
  "campaigns.campaign_deletion_will_be_implemented": "Campaign deletion will be implemented soon",
  "success.faq_added_successfully": "FAQ added successfully",
  "success.faq_updated_successfully": "FAQ updated successfully",
  "success.faq_deleted_successfully": "FAQ deleted successfully",
  "success.faq_deleted_successfully_1": "FAQ deleted successfully",
  "translations.translation_filter_applied": "Translation filter applied",
  "groups.group_filter_will_be_implemented": "Group filter will be implemented soon",
  "label.expected_format_namephoneemail_header_in": "Expected format: name,phone,email (header in first line)",
  "field_label.json_file": "JSON File:",
  "action_edit.faq_editing_function_not_implemented": "FAQ editing function not implemented",
  "general.preview_function_in_development": "Preview function in development",
  "success.grupo_atualizado_com_sucesso": "Group updated successfully",
  "success.group_created_successfully": "Group created successfully",
  "contacts.contact_group": "Contact Group:",
  "groups.target_group_optional": "Target Group (optional):",
  "success.group_deleted_successfully": "Group deleted successfully",
  "groups.grupo": "Group:",
  "success.grupos_recarregados_com_sucesso": "Groups reloaded successfully",
  "general.hours": "Hours"
};

/**
 * Initialize translations in database
 * Automatically populates with English as default language
 */
async function initializeTranslations() {
  let connection;
  try {
    connection = await pool.getConnection();
    
    console.log('🌍 Initializing translation system...');
    
    // 1. Set English as default active language
    await connection.execute(`
      INSERT INTO bot_settings (setting_key, setting_value) 
      VALUES ('active_language', 'en') 
      ON DUPLICATE KEY UPDATE setting_value = 'en'
    `);
    
    console.log('✅ Active language set: en (English)');
    
    // 2. Check if English translations already exist
    const [existing] = await connection.execute(
      'SELECT COUNT(*) as count FROM translations WHERE language_code = ?',
      ['en']
    );
    
    if (existing[0].count > 0) {
      console.log(`ℹ️  ${existing[0].count} English translations already exist in database`);
      
      // Check if there are new keys to add
      const [existingKeys] = await connection.execute(
        'SELECT translation_key FROM translations WHERE language_code = ?',
        ['en']
      );
      
      const existingKeySet = new Set(existingKeys.map(row => row.translation_key));
      const newKeys = Object.keys(defaultTranslations).filter(key => !existingKeySet.has(key));
      
      if (newKeys.length > 0) {
        console.log(`📝 Adding ${newKeys.length} new translation keys...`);
        
        for (const key of newKeys) {
          await connection.execute(`
            INSERT INTO translations (language_code, language_name, translation_key, translation_value) 
            VALUES (?, ?, ?, ?)
          `, ['en', 'English', key, defaultTranslations[key]]);
        }
        
        console.log(`✅ ${newKeys.length} new translations added`);
      } else {
        console.log('✅ All translations are already up to date');
      }
      
      return;
    }
    
    // 3. Populate English translations
    console.log(`📝 Populating ${Object.keys(defaultTranslations).length} English translations...`);
    
    let inserted = 0;
    for (const [key, value] of Object.entries(defaultTranslations)) {
      await connection.execute(`
        INSERT INTO translations (language_code, language_name, translation_key, translation_value) 
        VALUES (?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE translation_value = VALUES(translation_value)
      `, ['en', 'English', key, value]);
      inserted++;
    }
    
    console.log(`✅ ${inserted} English translations inserted successfully`);
    console.log('🎉 Translation system initialized!');
    
  } catch (error) {
    console.error('❌ Error initializing translations:', error);
    throw error;
  } finally {
    if (connection) connection.release();
  }
}

module.exports = { initializeTranslations };

